#delimit;
clear all;
set more off;
version 13;

/* Note: change Stata directory to the ../Stata/Code directory to run the code */
/*qui: cd "DIRECTORY IN HERE/Stata/Code"; */

* setting paths;
qui: do paths;


********************************************************************************
*
* CPI from Japan Statistics
*
********************************************************************************;

run japan_cpi_update;
use "$StDataPath/jpcpi", clear;
keep date cpi;
tempfile cpi;
save `cpi', replace;


********************************************************************************
*
* QUANTILES BY INCOME. Import data from Stats Japan. Downloaded from Japanese version of the website.
*
********************************************************************************;


import excel "$DataPath/Expenditure/income_expenditure_inc_quantiles_ts_allhouseholds.xlsx", cellrange(C16:TB941) sheet("Sheet2") firstrow clear; 

/* removing blank characters */
#delimit;
ds, has(type string);
foreach var of varlist `r(varlist)' {;
	qui replace `var'="";
};
qui destring _all, replace;

/* separating type */
qui gen date = ym(year,month);
qui gen date_type = date*100+quintile;
qui keep date_type *_all *_worker;

unab vars: *_all ;
local stubs: subinstr local vars "_all" "", all;
reshape long `stubs', i(date_type) j(hh_type) string;

/* creating date and type variable */
qui gen date = floor(date_type/100);
qui gen quintile = date_type-date*100;
qui drop date_type;

/* categorize workers */
qui gen highlow=(quintile>=4);
qui egen hh_type_cat = group(hh_type highlow);

/* collapse quintiles */
collapse (sum) `stubs' [aw=sample], by(date hh_type_cat);

/* label categories */
label define hh_type_cat_lab 1 "2-person HH Low-Income" 2 "2-person HH High-Income"  3 "Worker-HH Low-Income" 4 "Worker-HH High-Income" ;
label values hh_type_cat hh_type_cat_lab ;

/* merging cpi */
merge m:1 date using `cpi', nogenerate;
qui drop if hh_type_cat==.;
sort hh_type_cat date;

/* tsseting date */
tsset hh_type_cat date, monthly;
save "$StDataPath/expenditure_survey_income.dta", replace;


********************************************************************************
*
* CONSUMPTION BY AGE DISTRIBUTION. Import data from Stats Japan. Downloaded from Japanese version of the website.
*
********************************************************************************;


import excel "$DataPath/Expenditure/income_expenditure_age_ts_allhouseholds.xlsx", cellrange(C16:TB1919) firstrow clear; 


/* removing blank characters */
#delimit;
ds, has(type string);
foreach var of varlist `r(varlist)' {;
	qui replace `var'="";
};
qui destring _all, replace;

/* separating type */
qui gen date = ym(year,month);
qui gen date_type = date*100+maxage;
qui keep date_type *_all *_worker;

unab vars: *_all ;
local stubs: subinstr local vars "_all" "", all;
reshape long `stubs', i(date_type) j(hh_type) string;

/* creating date and type variable */
qui gen date = floor(date_type/100);
qui gen maxage = date_type-date*100;
qui drop date_type;

/* categorize workers */
qui gen highlow=(maxage>=60) if hh_type=="_all";
qui replace highlow=(maxage>=55) if hh_type=="_worker";
qui egen hh_type_cat = group(hh_type highlow);

/* collapse quintiles */
collapse (sum) `stubs' [aw=sample], by(date hh_type_cat);

/* label categories */
label define hh_type_cat_lab 1 "2-person HH Age<=59" 2 "2-person HH Age>59"  3 "Worker-HH Age<=54" 4 "Worker-HH Age>55" ;
label values hh_type_cat hh_type_cat_lab ;

/* merging cpi */
merge m:1 date using `cpi', nogenerate;
qui drop if hh_type_cat==.;
sort hh_type_cat date;

/* tsseting date */
tsset hh_type_cat date, monthly;
save "$StDataPath/expenditure_survey_age.dta", replace;

********************************************************************************
*
* CONSUMPTION BY OWNERSHIP DISTRIBUTION. Import data from Stats Japan. Downloaded from Japanese version of the website.
*
********************************************************************************;


import excel "$DataPath/Expenditure/income_expenditure_dwelling_ts_allhouseholds.xlsx", cellrange(C16:TB941) firstrow clear; 

/* removing blank characters */
#delimit;
ds, has(type string);
foreach var of varlist `r(varlist)' {;
	qui replace `var'="" if `var'=="***" | `var'=="-";
};
qui destring _all, replace;

/* separating type */
qui gen date = ym(year,month);
qui gen date_type = date*100+dwelling;
qui keep date_type *_all *_worker;

unab vars: *_all ;
local stubs: subinstr local vars "_all" "", all;
reshape long `stubs', i(date_type) j(hh_type) string;

/* creating date and type variable */
qui gen date = floor(date_type/100);
qui gen dwelling = date_type-date*100;
qui drop date_type;

/* categorize workers */
qui gen highlow=dwelling if dwelling<=3;
qui replace highlow=3 if dwelling>=3;
qui egen hh_type_cat = group(hh_type highlow);

/* collapse quintiles */
collapse (sum) `stubs' [aw=sample], by(date hh_type_cat);

/* label categories */
label define hh_type_cat_lab 1 "2-person Owner (inc Mortgager)" 2 "2-person Mortgager"  3 "2-person Renter" 4 "Worker-HH Owner (inc Mortgager)" 5 "Worker-HH Mortgager" 6 "Worker-HH Renter" ;
label values hh_type_cat hh_type_cat_lab ;

/* merging cpi */
merge m:1 date using `cpi', nogenerate;
qui drop if hh_type_cat==.;
sort hh_type_cat date;

/* tsseting date */
tsset hh_type_cat date, monthly;
save "$StDataPath/expenditure_survey_dwelling.dta", replace;


********************************************************************************
*
* Statistics by quintile
*
********************************************************************************;

local cutset income age dwelling;

foreach cut in `cutset' {;

	qui use  "$StDataPath/expenditure_survey_`cut'.dta", clear;
	
	local yearlist 2011 2012 2013;
	local varlist receipts income_regular withdrawals loans_lumpsum cons_exp food housing hh_durables dom_nondurables clothing trans_private savings disp_income;
	
	/* creating logs of series */
	foreach var of varlist `varlist' {;
			qui gen l`var' = ln(`var'/cpi);
			qui gen l`var'_sa = .;
			qui gen l`var'_sa_ma3 = .;
			qui gen l`var'_sa_ma5 = .;
			qui gen d12l`var' = S12.l`var';
	};
	
	/* filter series through a seasonal adjustment */
	qui qui levelsof hh_type_cat, local(hh_type_cat_no);
	gen month_dum = month(dofm(date));
	
	foreach quintile in `hh_type_cat_no' {;
		display "`quintile'";
		foreach var of varlist `varlist' {;
			di "`var'";
			qui su l`var' if hh_type_cat==`quintile';
			if `r(N)'>0 {;
				qui xi, noomit prefix(_) i.month_dum;
				capture qui arima l`var' _month_dum_* if hh_type_cat==`quintile', ar(1) ma(1) noconstant;
				if _rc!=0 {;
					capture qui arima l`var' _month_dum_* if hh_type_cat==`quintile', ar(1) ma(2) noconstant;
				};
				
				/*Subtract month/quarter dummy from SA variable*/
				local unitmean = 0;
				forvalues mm = 1/12 {;
					qui replace l`var'_sa = l`var'  - _b[_month_dum_`mm'] if month_dum==`mm' &  hh_type_cat==`quintile'; 
					local unitmean = `unitmean' + _b[_month_dum_`mm'];
				}; 
				local unitmean = `unitmean'/12;  
				/*Add back mean of month/quarter dummies to SA variable*/
				qui replace l`var'_sa = l`var'_sa + `unitmean' if hh_type_cat==`quintile'; 
				
				foreach smoothlen in 3 5 {;
					local smoother = (`smoothlen'-1)/2;
					qui tssmooth ma l`var'_sa_ma`smoothlen'_ = l`var'_sa if hh_type_cat==`quintile', window(`smoother' 1 `smoother');
					qui replace l`var'_sa_ma`smoothlen' = l`var'_sa_ma`smoothlen'_ if hh_type_cat==`quintile'; 
					qui drop l`var'_sa_ma`smoothlen'_;
				};
			};
		};
	};
	
	/* creating baseline values for variables of interest */
	foreach year in `yearlist' {;
		foreach oldvar in `varlist' {;
		
			local var l`oldvar'_sa;
			
			qui egen `var'_`year'_ = mean(`var') if yofd(dofm(date))==`year', by(hh_type_cat);
			qui egen `var'_`year' = min(`var'_`year'_), by(hh_type_cat);
			qui drop `var'_`year'_;
			
			
			qui gen d`var'_`year' = `var' - `var'_`year';
			foreach smoothlen in 3 5 {;
				qui gen d`var'_ma`smoothlen'_`year' = `var'_ma`smoothlen' - `var'_`year';
			};
			
			
		};
	};
	
	/* saving data */
	save "$StDataPath/expenditure_survey_`cut'_mod.dta", replace;
};

	********************************************************************************
	*
	* Plotting outcomes
	*
	********************************************************************************;
	# delimit;
	local varlist receipts income_regular withdrawals loans_lumpsum cons_exp food housing hh_durables dom_nondurables clothing trans_private savings disp_income;
	local labellist `"Receipts"' `"Regular Income"' `"Withdrawals"' `"Lump-sum loans"' `"Consumption Expenditures"' `"Food Expenditures"' `"Housing Expenditures"' `"Household Durables"' `"Domestic Nondurables"' `"Clothing Expenditures"' `"Private Transportation"' `"Savings"' `"Disposable Income"';
	
	local plotsetincome "plot1(lwidth(medthick) lcolor(blue) lpattern(solid)) plot2(lwidth(medthick) lcolor(red) lpattern(dash)) plot3(lwidth(medthick) lcolor(green) lpattern(solid)) plot4(lwidth(medthick) lcolor(black) lpattern(dash))  " ;
	local plotsetage "`plotsetincome'";
	local plotsetdwelling "plot1(lwidth(medthick) lcolor(blue) lpattern(solid)) plot2(lwidth(medthick) lcolor(gray) lpattern(dash_dot)) plot3(lwidth(medthick) lcolor(green) lpattern(solid)) plot4(lwidth(medthick) lcolor(orange) lpattern(dash)) plot5(lwidth(medthick) lcolor(black) lpattern(dash_dot))  " ;
	
foreach cut in `cutset' {;

	use "$StDataPath/expenditure_survey_`cut'_mod.dta", clear;
	
	forvalues jj=1(1)`=wordcount("`varlist'")' {;
		local oldvar : word `jj' of `varlist';
		local varlabel : word `jj' of `"`labellist'"';
	
		qui replace dl`oldvar'_sa_2011=dl`oldvar'_sa_2011*100;
		*qui replace dl`oldvar'_sa_2011=0 if date==ym(2011,1) & dl`oldvar'_sa_2011==.;
		
		/* standard graph */
		xtline dl`oldvar'_sa_2011 if tin(2011m1,2015m5), overlay i(hh_type_cat) t(date) name(dl`oldvar'_`cut') ///
		legend(cols(2) ring(0) position(6)) ///
		`plotset`cut'' ///
		title("") ytitle("% deviation relative to 2011 average") ttitle("") tline(2012m11) plotregion(style(none)) ///
		graphregion(color(white)) tlabel(2011m1 2012m1 2013m1 2014m1 2015m1, format(%tm));
		graph export "$StGrphPath/cons_exp_`oldvar'_`cut'.pdf", replace;	

		
		/* 5-month moving average graph */
		qui replace dl`oldvar'_sa_ma5_2011=dl`oldvar'_sa_ma5_2011*100;
		xtline dl`oldvar'_sa_ma5_2011 if tin(2011m1,2015m5), overlay i(hh_type_cat) t(date) name(dl`oldvar'_m5_`cut') ///
		legend(cols(2) ring(0) position(6)) ///
		`plotset`cut'' ///
		title("") ytitle("% deviation relative to 2011 average (MA5)") ttitle("") tline(2012m11) plotregion(style(none)) ///
		graphregion(color(white)) tlabel(2011m1 2012m1 2013m1 2014m1 2015m1, format(%tm));
		graph export "$StGrphPath/cons_exp_`oldvar'_ma5_`cut'.pdf", replace;
	 
	 
	};
};

/*
********************************************************************************
*
* Presentation graph
*
********************************************************************************;

foreach cut in dwelling {;

	use "$StDataPath/expenditure_survey_`cut'_mod.dta", clear;
	
	forvalues jj=9(1)9 {;
		local oldvar : word `jj' of `varlist';
		local varlabel : word `jj' of `"`labellist'"';
	
		qui replace dl`oldvar'_sa_2011=dl`oldvar'_sa_2011*100;
		*qui replace dl`oldvar'_sa_2011=0 if date==ym(2011,1) & dl`oldvar'_sa_2011==.;
		
		xtline dl`oldvar'_sa_2011 if tin(2011m1,2015m5), overlay i(hh_type_cat) t(date) name(dl`oldvar'_`cut') ///
		legend(cols(2) ring(0) position(12)) ytick(-20(20)65) ///
		`plotset`cut'' ///
		title("") ytitle("% deviation relative to 2011 average") ttitle("") tline(2012m11) plotregion(style(none)) ///
		graphregion(color(white)) tlabel(2011m1 2012m1 2013m1 2014m1 2015m1, format(%tm));
		graph export "$StGrphPath/cons_exp_`oldvar'_`cut'.pdf", replace;	


	};
};
