***********************
* Early Impact of ACA *
* Amanda Kowalski     *
*                     *
* 13 October 2014     *
***********************

**************************************************************
* PURPOSE: Compute miscellaneous statistics for use in paper *
**************************************************************

clear
set matsize 2000
set type double
set more off, permanently

local tag "misc"
capture log close `tag'
log using "logs/`tag'.log", replace name(`tag')

**************************************
* Miscellaneous data characteristics *
**************************************

use "data/intermediate/quarterly_data_nd_imputed.dta", clear

assert enr != . & ///
	   mmonths != . & ///
	   premium != . & ///
	   cost != . & ///
	   enr_imp != . & ///
	   mmonths_imp != . & ///
	   premium_imp != . & ///
	   cost_imp != .
	   
*Drop 23 "null" observations -- have all 0s for enr, mmonths, premium, cost
drop if enr == 0 & ///
	    mmonths == 0 & ///
	    premium == 0 & ///
	    cost == 0 & ///
	    enr_imp == 0 & ///
	    mmonths_imp == 0 & ///
	    premium_imp == 0 & ///
	    cost_imp == 0

*Number of firms in the data
codebook id // 393 unique values
count // 6727 distinct id-yq obs
di 393 * 26 // 10218

use "data/intermediate/quarterly_data_nd_imputed.dta", clear

*Number of observations with reporting issues (skip quarters)
gen flag1 = mmonths == 0 & yr <= 2011
gen flag2 = mmonths == 0 & yr == 2013
bysort id : egen flag1_mm = max(flag1)
bysort id : egen flag2_mm = max(flag2)

su mmonths if flag1_mm == 1 & yr <= 2011
su mmonths if yr <= 2011				// .05937675

su mmonths if flag2_mm == 1 & yr == 2013
su mmonths if yr == 2013				// .00203982

*Firms operating in 2013Q4 but not in 2014
gen is_2013q4 = yq == yq(2013,4)
gen is_2014q1 = yq == yq(2014,1)
bysort id : egen has_2013q4 = max(is_2013q4)
bysort id : egen has_2014q1 = max(is_2014q1)

su mmonths if has_2013q4 == 1 & has_2014q1 == 0 & yr == 2013
su mmonths if yr == 2013				// .00927932

***********************************************
/* Relationship between raw and imputed data */
***********************************************

*Enrollment
count if enr != enr_imp					// 438 obs
correl enr enr_imp					// 0.9963
su enr if enr != enr_imp
su enr							// .04724633
su enr_imp if enr != enr_imp
su enr_imp						// .05573786

*Member Months
count if mmonths != mmonths_imp 			// 492 obs
correl mmonths mmonths_imp				// 0.9725
su mmonths if mmonths != mmonths_imp
su mmonths						// .06811607
su mmonths_imp if mmonths != mmonths_imp
su mmonths_imp						// .06815272

*Premiums
count if premium != premium_imp 			// 579 obs
correl premium premium_imp				// 0.9917
su premium if premium != premium_imp
su premium						// .05661948
su premium_imp if premium != premium_imp
su premium_imp						// .06825262

*Costs
count if cost != cost_imp				// 676 obs
correl cost cost_imp					// 0.9729
su cost if cost != cost_imp
su cost							// .06203164
su cost_imp if cost != cost_imp
su cost_imp						// .0794268

*STATISTICS ON DATA ALLOCATED*
use "data/intermediate/data_allocated.dta", clear

gen idyq = id + yrqtr
encode state, gen(code)

foreach var in "e" "mm" "p" "c" {
	replace `var'_imp = 0 if `var'_imp == .
}

*Unique allocation by state
bysort idyq : egen count = count(code) if e_imp > 0 | mm_imp > 0 | p_imp > 0 | c_imp > 0
bysort idyq : egen ct = max(count)
drop count
codebook idyq if ct == 1 // 5728 idyq uniquely allocated to states
su *_imp if ct == 1
su *_imp

*Enrollment
// .77053834 uniquely assigned

*Member Months
// .77296708 uniquely assigned

*Premiums
// .7816662 uniquely assigned

*Costs
// .79128117 uniquely assigned

******************************
* Firm statistics (by state) *
******************************

*Merge data on firms with plans on exchanges
merge m:1 id using "data/intermediate/firms_exch.dta"
assert _merge != 1
keep if _merge == 3
drop _merge

rename firm_exch exch
rename HMO hmo
foreach var in "e" "e_imp" "mm" "mm_imp" {
	gen hmo_`var'  = hmo  * `var'
	gen exch_`var' = exch * `var'
}

*Obtain firm count by state-quarter
gen firm_ct = mm_imp > 0 & mm_imp != .

*Collapse to state-level
collapse (sum) e e_imp exch_e exch_e_imp hmo_e hmo_e_imp ///
	       mm mm_imp exch_mm exch_mm_imp hmo_mm hmo_mm_imp ///
	       p p_imp c c_imp firm_ct, by(state yrqtr yq yr qtr)
order state yrqtr yq yr qtr

*Drop if all values 0 or missing (should only drop California 2008Q1-Q4)
drop if (e == 0 | mm == 0 | p == 0 | c == 0) & ///
	(e_imp == 0 | mm_imp == 0 | p_imp == 0 | c_imp == 0)

replace e = e / 10^3
replace e_imp = e_imp / 10^3
replace exch_e = exch_e / 10^3
replace exch_e_imp = exch_e_imp / 10^3
replace hmo_e = hmo_e / 10^3
replace hmo_e_imp = hmo_e_imp / 10^3

replace mm = mm / 10^3
replace mm_imp = mm_imp / 10^3
replace exch_mm = exch_mm / 10^3
replace exch_mm_imp = exch_mm_imp / 10^3
replace hmo_mm = hmo_mm / 10^3
replace hmo_mm_imp = hmo_mm_imp / 10^3

gen P = p / mm
gen P_imp = p_imp / mm_imp
gen C = c / mm
gen C_imp = c_imp / mm_imp

*Label variables
label var e "Enr."
label var e_imp "Enr. (imp)"
label var exch_e "Exchange Enr."
label var exch_e_imp "Exchange Enr. (imp)"
label var hmo_e "HMO Enr."
label var hmo_e_imp "HMO Enr. (imp)"
label var mm "MM"
label var mm_imp "MM (imp)"
label var exch_mm "Exchange MM"
label var exch_mm_imp "Exchange MM (imp)"
label var hmo_mm "HMO MM"
label var hmo_mm_imp "HMO MM (imp)"
label var P "P"
label var P_imp "P (imp)"
label var C "C"
label var C_imp "C (imp)"
label var firm_ct "No. Firms Operating"
label var yq "Year-Quarter"

encode state, gen(state_c)

*Order data
sort state state_c yq
order state state_c yq

compress

*Set panel data
xtset state_c yq

*********************
/* FIRM STATISTICS */
*********************
keep if yq >= yq(2013,2)
keep state yrqtr exch_mm_imp hmo_mm_imp firm_ct
replace exch_mm_imp = exch_mm_imp / 3
replace hmo_mm_imp  = hmo_mm_imp  / 3
rename exch_mm_imp exch_enr
rename hmo_mm_imp  hmo_enr
reshape wide exch_enr hmo_enr firm_ct, i(state) j(yrqtr) string

order state exch* hmo* firm*

outsheet using "output/tables/firm_stats.csv", comma names replace

keep state firm_ct2013Q3
rename firm_ct2013Q3 firm_ct
label var firm_ct "Number of Insurers 2013Q3"
save "data/intermediate/firm_ct.dta", replace

log close `tag'
