%  irf_comp.m

% comparison of IRF of UE to monetary shocks (Romer Romer) and financial
% shocks (Gilchrist Zachrajsek).

% Compare half-lives of the responses to RR and GZ shocks

clear all
close all

%=========================================================================
%  replicate GZ results and generate IRF of UE
GZdata;                % loads quarterly data in GZ from 1973Q1-2010Q3
ebp=GZ(:,1);           % excess bond premium
C=GZ(:,2);             % real consumption
I=GZ(:,3);             % real business investment
Y=GZ(:,4);             % real GDP
P=GZ(:,5);             % GDP imp. price deflator
XS=GZ(:,6);            % excess stock return
TB=GZ(:,7);            % 10-year Treasury rate
FFR=GZ(:,8);           % effective FFR

UE_q;                  % load quarterly UE from 1968Q1-2010Q3.

% first difference relevant variables
dC=100*(log(C(2:length(ebp),1))-log(C(1:length(ebp)-1)));
dI=100*(log(I(2:length(ebp),1))-log(I(1:length(ebp)-1)));
dY=100*(log(Y(2:length(ebp),1))-log(Y(1:length(ebp)-1)));
dP=100*(log(P(2:length(ebp),1))-log(P(1:length(ebp)-1)));

% make lags for VAR
dC=makelags(dC,2); dI=makelags(dI,2); dY=makelags(dY,2); dP=makelags(dP,2); 
ebp=makelags(ebp(2:length(ebp)),2); TB=makelags(TB(2:length(TB)),2); XS=makelags(XS(2:length(XS)),2); FFR=makelags(FFR(2:length(FFR)),2);
UE=makelags(ue(22:length(ue)),2);
due=ue(2:length(ue))-ue(1:length(ue)-1);
dUE=makelags(due(21:length(due)),2);

% run VAR
dat=[dC dI dY dP ebp XS TB FFR];
options.irfstd=1;
out_GZ=varcg(dat,2,options);     

dat_UE=[dC dI UE dP ebp XS TB FFR];
options.irfstd=1;
out_UE=varcg(dat_UE,2,options);     
[HLGZ_mean, HLGZ_p50, HLGZ_sd , HLGZ_p5, HLGZ_p95]   =  varcg_halflife(dat_UE,2,options);  


dat_dUE=[dC dI dUE dP ebp XS TB FFR];
options.irfstd=1; options.Nirfstd=100;
out_dUE=varcg(dat_dUE,2,options);     

%==========================================================================
% now estimate IRF of UE to Romer and Romer shock (extended sample, monthly)
USdata;                     % load UE and rrsh (UE rate and Romer Romer mp shocks updated to Dec. 2008).
ARlag=24;
MAlag=36;
T=60;
maxlag=max(ARlag,MAlag);
UE=makelags(UE,maxlag);
rrsh=makelags(rrsh,maxlag);
[HL_mean, HL_p50, HL_sd , HL_p5, HL_p95] =se_halflife(UE(:,1),[ones(length(UE),1) UE(:,2:ARlag+1) rrsh(:,2:MAlag+1)],T,ARlag+2,100,1) ;

RR_HL_mean=HL_mean/3;
RR_HL_p50=HL_p50/3;
RR_HL_sd=HL_sd/3;
RR_HL_p5=HL_p5/3;
RR_HL_p95=HL_p95/3;

%=========================================================================
% now estimate IRF of UE to Romer and Romer shock (ORIGINAL sample, monthly)
MPshockdata;                        % load macro data: IP, UE, CPI, CPIc, PPI, FFR, PCOM: 1968:1-1996:12, monthly
RRshock;                            % load MP shock measure from Romer and Romer (2004): uRR 1969:1-1996:12
UE2=makelags(UE,60);
uRR2=makelags([zeros(12,1); uRR],60);

[HL_mean, HL_p50, HL_sd , HL_p5, HL_p95] =se_halflife(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,100,1) ;


RRO_HL_mean=HL_mean/3;
RRO_HL_p50=HL_p50/3;
RRO_HL_sd=HL_sd/3;
RRO_HL_p5=HL_p5/3;
RRO_HL_p95=HL_p95/3;


%=======================================================================
%                   display results
%=======================================================================

disp(['                       GZ RR RR-original']);
disp(['mean     ' num2str(HLGZ_mean) '  ' num2str(RR_HL_mean) '  ' num2str(RRO_HL_mean)])
disp(['median   ' num2str(HLGZ_p50) '  ' num2str(RR_HL_p50) '  ' num2str(RRO_HL_p50)])
disp(['std      ' num2str(HLGZ_sd) '  ' num2str(RR_HL_sd) '  ' num2str(RRO_HL_sd)])
disp(['p05      ' num2str(HLGZ_p5) '  ' num2str(RR_HL_p5) '  ' num2str(RRO_HL_p5)])
disp(['p95      ' num2str(HLGZ_p95) '  ' num2str(RR_HL_p95) '  ' num2str(RRO_HL_p95)])
