%  irf_comp.m

% comparison of IRF of UE to monetary shocks (Romer Romer) and financial
% shocks (Gilchrist Zachrajsek).

clear all
close all

%=========================================================================
%  replicate GZ results and generate IRF of UE
GZdata;                % loads quarterly data in GZ from 1973Q1-2010Q3
ebp=GZ(:,1);           % excess bond premium
C=GZ(:,2);             % real consumption
I=GZ(:,3);             % real business investment
Y=GZ(:,4);             % real GDP
P=GZ(:,5);             % GDP imp. price deflator
XS=GZ(:,6);            % excess stock return
TB=GZ(:,7);            % 10-year Treasury rate
FFR=GZ(:,8);           % effective FFR

UE_q;                  % load quarterly UE from 1968Q1-2010Q3.

% first difference relevant variables
dC=100*(log(C(2:length(ebp),1))-log(C(1:length(ebp)-1)));
dI=100*(log(I(2:length(ebp),1))-log(I(1:length(ebp)-1)));
dY=100*(log(Y(2:length(ebp),1))-log(Y(1:length(ebp)-1)));
dP=100*(log(P(2:length(ebp),1))-log(P(1:length(ebp)-1)));

% make lags for VAR
dC=makelags(dC,2); dI=makelags(dI,2); dY=makelags(dY,2); dP=makelags(dP,2); 
ebp=makelags(ebp(2:length(ebp)),2); TB=makelags(TB(2:length(TB)),2); XS=makelags(XS(2:length(XS)),2); FFR=makelags(FFR(2:length(FFR)),2);
UE=makelags(ue(22:length(ue)),2);
due=ue(2:length(ue))-ue(1:length(ue)-1);
dUE=makelags(due(21:length(due)),2);

% run VAR
dat=[dC dI dY dP ebp XS TB FFR];
options.irfstd=1;
out_GZ=varcg(dat,2,options);     

dat_UE=[dC dI UE dP ebp XS TB FFR];
options.irfstd=1;
out_UE=varcg(dat_UE,2,options);     

dat_dUE=[dC dI dUE dP ebp XS TB FFR];
options.irfstd=1; options.Nirfstd=100;
out_dUE=varcg(dat_dUE,2,options);     

% plot responses of RGDP and UE from GZ replication
%figure(1)
%irf_dY=out_GZ.irf(1:20,3,5);
%irf_Y(1)=irf_dY(1); for j=2:length(irf_dY) irf_Y(j)=irf_Y(j-1)+irf_dY(j); end
%subplot(3,1,1)
%plot(irf_Y); title('RGDP');
%subplot(3,1,2)
%plot(out_UE.irf(1:20,3,5)); title('Unemployment (levels in VAR)');
%irf_dUE=out_dUE.irf(1:20,3,5);
%irf_UE(1)=irf_dUE(1); for j=2:length(irf_dUE) irf_UE(j)=irf_UE(j-1)+irf_dUE(j); end
%subplot(3,1,3)
%plot(irf_UE); title('Unemployment (differences in VAR)');
%clear UE

%==========================================================================
% now estimate IRF of UE to Romer and Romer shock (extended sample, monthly)
USdata;                     % load UE and rrsh (UE rate and Romer Romer mp shocks updated to Dec. 2008).
ARlag=24;
MAlag=36;
T=60;
maxlag=max(ARlag,MAlag);
UE=makelags(UE,maxlag);
rrsh=makelags(rrsh,maxlag);
irf=impulse(UE(:,1),[ones(length(UE),1) UE(:,2:ARlag+1) rrsh(:,2:MAlag+1)],T,ARlag+2,1) ;
sd=se(UE(:,1),[ones(length(UE),1) UE(:,2:ARlag+1) rrsh(:,2:MAlag+1)],T,ARlag+2,100,1) ;
irf_UB=irf+sd;
irf_LB=irf-sd;

% make RR responses quarterly
ind=1;
for j=1:20
    irfRR(j)=mean(irf(ind:ind+2));
    irfUB(j)=mean(irf_UB(ind:ind+2));
    irfLB(j)=mean(irf_LB(ind:ind+2));
    ind=ind+3;
end

% estimate GZ stopping in Dec. 2008
%dat_2008=dat_UE(1:141,:);
%options.irfstd=1;
%out_UE=varcg(dat_UE,2,options);     


% scaling factor so IRF of UE is same scale
scale=max(out_UE.irf(1:20,3,5))/max(irfRR);

%figure(2)
%plot(1:20,out_UE.irf(1:20,3,5),'k','Linewidth',2); hold on
%plot(1:20,scale*irfRR,'b','Linewidth',2);
%plot(1:20,out_UE.irf(1:20,3,5)+out_UE.irfstd(1:20,3,5),'k:','Linewidth',1);
%plot(1:20,out_UE.irf(1:20,3,5)-out_UE.irfstd(1:20,3,5),'k:','Linewidth',1);
%plot(1:20,scale*irfUB,'b:','Linewidth',1);
%plot(1:20,scale*irfLB,'b:','Linewidth',1);


%=========================================================================
% now estimate IRF of UE to Romer and Romer shock (ORIGINAL sample, monthly)
MPshockdata;                        % load macro data: IP, UE, CPI, CPIc, PPI, FFR, PCOM: 1968:1-1996:12, monthly
RRshock;                            % load MP shock measure from Romer and Romer (2004): uRR 1969:1-1996:12
UE2=makelags(UE,60);
uRR2=makelags([zeros(12,1); uRR],60);

irf=impulse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,1) ;
sd=se(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,100,1) ;
irf_UB=irf+1.64*sd;
irf_LB=irf-1.64*sd;

% make RR responses quarterly
ind=1;
for j=1:20
    irfRR(j)=mean(irf(ind:ind+2));
    irfUB(j)=mean(irf_UB(ind:ind+2));
    irfLB(j)=mean(irf_LB(ind:ind+2));
    ind=ind+3;
end

% scaling factor so IRF of UE is same scale
scale=max(out_UE.irf(1:20,3,5))/max(irfRR);

%figure(3)
%plot(1:20,out_UE.irf(1:20,3,5),'k','Linewidth',2); hold on
%plot(1:20,scale*irfRR,'b','Linewidth',2);
%plot(1:20,out_UE.irf(1:20,3,5)+1.64*out_UE.irfstd(1:20,3,5),'k:','Linewidth',1);
%plot(1:20,out_UE.irf(1:20,3,5)-1.64*out_UE.irfstd(1:20,3,5),'k:','Linewidth',1);
%plot(1:20,scale*irfUB,'b:','Linewidth',1);
%plot(1:20,scale*irfLB,'b:','Linewidth',1);

%figure(20)
%plot(1:22,[nan(2,1); out_UE.irf(1:20,3,5)],'k','Linewidth',2); hold on
%plot(1:22,[scale*irfRR nan(1,2)],'b','Linewidth',2);
%plot(1:22,[nan(2,1); out_UE.irf(1:20,3,5)+out_UE.irfstd(1:20,3,5)],'k:','Linewidth',1);
%plot(1:22,[nan(2,1); out_UE.irf(1:20,3,5)-out_UE.irfstd(1:20,3,5)],'k:','Linewidth',1);
%plot(1:22,[scale*irfUB nan(1,2)],'b:','Linewidth',1);
%plot(1:22,[scale*irfLB nan(1,2)],'b:','Linewidth',1);

%=======================================================================
%  construct estimates of UE responses "after peak" and half-lives
%  for both RR 2004 and GZ 2012.
N=2000;                              % number of simulations
MPshockdata;                        % load macro data: IP, UE, CPI, CPIc, PPI, FFR, PCOM: 1968:1-1996:12, monthly
RRshock;                            % load MP shock measure from Romer and Romer (2004): uRR 1969:1-1996:12
UE2=makelags(UE,60);
uRR2=makelags([zeros(12,1); uRR],60);

% Romer and Romer monthly responses after peak UE
irf=peakresponse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,1) ;   
dist=dist_peakresponse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,N+1,1) ;
dist_sort=sort(dist);

% make quarterly
dist_q=ones(rows(dist),1);
for j=2:18
    dist_q(:,j)=mean(dist_sort(:,3*(j-2)+2:3*(j-2)+4)')';
end

% GZ quarterly responses after peak UE
%dat_UE=[dC dI UE dP ebp XS TB FFR];
options.Nirfstd=N; options.var=3; options.shock=5;
out_gz1=varcg_peak(dat_UE,2,options);     
dist_gz=out_gz1.impdist;
dist_gz=sort(dist_gz);
dist_gz=dist_gz(:,1:cols(dist_q));

% plot the two distributions of UE responses after peak.
figure(81)
%subplot(1,2,1)
plot(0:cols(dist_q)-1,dist_q(floor(0.5*rows(dist)),:),'k','Linewidth',2); hold on
plot(0:cols(dist_q)-1,dist_q(floor(0.95*rows(dist)),:),'k--','Linewidth',2);
plot(0:cols(dist_q)-1,dist_gz(floor(0.5*rows(dist)),:),'bo-','Linewidth',2,'MarkerFaceColor','white'); 
xv = [(0:cols(dist_q)-1),fliplr((0:cols(dist_q)-1))];
yv = [dist_gz(floor(0.95*rows(dist)),:),fliplr(dist_gz(floor(0.05*rows(dist)),:))];
hReg = fill(xv,yv,[0.85 0.85 0.85],'EdgeColor','none'); hold on
plot(0:cols(dist_q)-1,dist_q(floor(0.5*rows(dist)),:),'k','Linewidth',2); hold on
plot(0:cols(dist_q)-1,dist_gz(floor(0.5*rows(dist)),:),'bo-','Linewidth',2,'MarkerFaceColor','white'); 
plot(0:cols(dist_q)-1,dist_q(floor(0.95*rows(dist)),:),'k--','Linewidth',2);
legend('UE Persistence after Monetary Policy Shock','UE Persistence after Monetary Policy Shock (90% CI)','UE Persistence after Financial Shock','UE Persistence after Financial Shock (90% CI)');
plot(0:cols(dist_q)-1,dist_q(floor(0.05*rows(dist)),:),'k--','Linewidth',2);
%plot(0:cols(dist_q)-1,dist_gz(floor(0.95*rows(dist)),:),'b:','Linewidth',2);
%plot(0:cols(dist_q)-1,dist_gz(floor(0.05*rows(dist)),:),'b:','Linewidth',2);
plot(0:cols(dist_q)-1,zeros(1,cols(dist_q)),'k','Linewidth',1);
xlabel('Quarters after Peak Effect of Shock on Unemployment');
ylabel('Unemployment relative to Peak Rise');
xlim([0 12]);
ylim([-.8 1]);

% plot equivalent dynamics for UE gaps after typical recession and Great Recession
UE_persistencedata;
figure(82)
%subplot(1,2,2)
plot(0:12,Ppre90,'ro-','Linewidth',2,'MarkerFaceColor','white'); hold on
plot(0:12,P2007,'k','Linewidth',2); 
xv = [(0:12),fliplr(0:12)];
yv = [dist_gz(floor(0.95*rows(dist)),1:13),fliplr(dist_gz(floor(0.05*rows(dist)),1:13))];
hReg = fill(xv,yv,[0.85 0.85 0.85],'EdgeColor','none'); hold on
plot(0:12,Ppre90,'ro-','Linewidth',2,'MarkerFaceColor','white'); hold on
plot(0:12,P2007,'k','Linewidth',2); 
%plot(0:12,dist_gz(floor(0.95*rows(dist)),1:13),'k:','Linewidth',2);
%plot(0:12,dist_gz(floor(0.05*rows(dist)),1:13),'k:','Linewidth',2);
plot(0:12,zeros(1,13),'k','Linewidth',1);
legend('UE Persistence in Average Pre-1990 Recessions','UE Persistence in 2007 Recession','UE Persistence after Financial Shock (90% CI)');
xlabel('Quarters after Peak Effect of Shock on Unemployment');
ylabel('Unemployment relative to Peak Rise');
ylim([-.8,1]);
xlim([0 12]);


%=========================================================================
%  now construct rise in UE predicted by financial shocks starting in 2007.

options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=134;                          % start shocks in 2007Q1
options.N=15;                               % feed in shocks until 2010Q3
options.T=26;                               % counterfactual until 2013Q2
out=varcg_counter(dat_UE,2,options);        % this constructs counterfactual
counter=[0 ; out.counter];
counter2010Q3=[nan(15,1); counter(16:length(counter))]; % counterfactuals starting 2010Q3
counterSHOCKS=[counter(1:16); nan(11,1)];

UE_2006q4;                                  % load quarterly UE (UEq06): 2006Q4-2013Q2, plus recession07 dummy variable (-100,100)
time=(2006.75:0.25:2013.25);

figure(73)
plot(time,(UEq06-UEq06(1))/max(UEq06-UEq06(1)),'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/max(counter),'b','Linewidth',2);
plot(time,counter2010Q3/max(counter),'b--','Linewidth',2);
%area(time68,recession68*100,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
area(time,recession07,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq06-UEq06(1))/max(UEq06-UEq06(1)),'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/max(counter),'b','Linewidth',2);
plot(time,counter2010Q3/max(counter),'b--','Linewidth',2);
ylim([-.1 1.3]); xlim([2006.75 2013.25]);
ylabel('Normalized Unemployment Rates');
legend('UE during 2007 Recession (relative to peak)','Predicted UE from Financial Shocks (relative to peak)');

scale=max(UEq06-UEq06(1));
figure(74)
subplot(1,2,2)
plot(time,(UEq06-UEq06(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/scale,'b','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counterSHOCKS)/scale,'r','Linewidth',1.5); hold on;
plot(time,counter2010Q3/scale,'b--','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counter)/scale,'r--','Linewidth',1.5); hold on;
%area(time68,recession68*100,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
area(time,recession07,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq06-UEq06(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/scale,'b','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counterSHOCKS)/scale,'r','Linewidth',1.5); hold on;
plot(time,counter2010Q3/scale,'b--','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counter)/scale,'r--','Linewidth',1.5); hold on;
ylim([-.5 1.5]); 
xlim([2006.75 2013.25]);
ylabel('Unemployment Rate');
legend('Actual (Normalized) UE','Predicted UE from Financial Shocks','Unexplained UE');

%========================================================================
% construct counterfactual UE paths from financial shocks for 1990, 2001
% and 2007 recessions.
%========================================================================

% 1990 recession
options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=65;                           % start shocks in 1989Q4
options.N=15;                               % feed in shocks through 1993Q4
options.T=26;                               % counterfactual until 1996Q1
out_90=varcg_counter(dat_UE,2,options);     % this constructs counterfactual
counter90=[0 ; out_90.counter];

UE_1989q3;                                  % load quarterly UE (UEq89): 1989Q3-1996Q1, plus recession90 dummy variable (-100,100)
time=(1989.5:0.25:1996);
scale=max(UEq89-UEq89(1));
figure(77)
%subplot(1,3,1)
plot(time,(UEq89-UEq89(1))/scale,'k','Linewidth',2); hold on;
plot(time,counter90/scale,'b','Linewidth',2);
plot(time,((UEq89-UEq89(1))-counter90)/scale,'r','Linewidth',2); hold on;
area(time,recession90,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq89-UEq89(1))/scale,'k','Linewidth',2); hold on;
plot(time,counter90/scale,'b','Linewidth',2);
plot(time,((UEq89-UEq89(1))-counter90)/scale,'r','Linewidth',2); hold on;
ylim([-.5 1.5]); 
xlim([1989.5 1996]);
ylabel('Unemployment Rates');
legend('UE during 2007 Recession','Predicted UE from Financial Shocks','Unexplained UE');

% 2001 recession
options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=107;                          % start shocks in 2000Q2
options.N=15;                               % feed in shocks through 2003Q4
options.T=26;                               % counterfactual until 2006Q3
out_01=varcg_counter(dat_UE,2,options);     % this constructs counterfactual
counter01=[0 ; out_01.counter];

UE_2000q1;                                  % load quarterly UE (UEq00): 2000Q1-2006Q3, plus recession01 dummy variable (-100,100)
time=(2000:0.25:2006.5);
scale=max(UEq00-UEq00(1));
figure(74)
subplot(1,2,1)
plot(time,(UEq00-UEq00(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counter01/scale,'b','Linewidth',2);
plot(time,((UEq00-UEq00(1))-counter01)/scale,'r','Linewidth',1.5); hold on;
area(time,recession01,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq00-UEq00(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counter01/scale,'b','Linewidth',4);
plot(time,((UEq00-UEq00(1))-counter01)/scale,'r','Linewidth',1.5); hold on;
ylim([-.5 1.5]); 
xlim([2000 2006.5]);
ylabel('Unemployment Rate');
sdf

%%% put all three recessions in one figure
figure(69)
subplot(1,3,1)
plot(time,(UEq89-UEq89(1)),'k','Linewidth',2); hold on;
plot(time,counter90,'b','Linewidth',2);
plot(time,(UEq89-UEq89(1))-counter90,'r','Linewidth',2); hold on;
area(time,recession90,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq89-UEq89(1)),'k','Linewidth',2); hold on;
plot(time,counter90,'b','Linewidth',2);
plot(time,(UEq89-UEq89(1))-counter90,'r','Linewidth',2); hold on;
ylim([-1 6]); 
xlim([1989.5 1996]);
ylabel('Unemployment Rates');

subplot(1,3,2)
plot(time,(UEq00-UEq00(1)),'k','Linewidth',2); hold on;
plot(time,counter01,'b','Linewidth',2);
plot(time,(UEq00-UEq00(1))-counter01,'r','Linewidth',2); hold on;
area(time,recession01,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq00-UEq00(1)),'k','Linewidth',2); hold on;
plot(time,counter01,'b','Linewidth',2);
plot(time,(UEq00-UEq00(1))-counter01,'r','Linewidth',2); hold on;
ylim([-1 6]); 
xlim([2000 2006.5]);
ylabel('Unemployment Rates');

subplot(1,3,3)
plot(time,(UEq06-UEq06(1)),'k','Linewidth',2); hold on;
plot(time,counterSHOCKS,'b','Linewidth',2);
plot(time,(UEq06-UEq06(1))-counter,'r','Linewidth',2); hold on;
plot(time,counter2010Q3,'b--','Linewidth',2);
area(time,recession07,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq06-UEq06(1)),'k','Linewidth',2); hold on;
plot(time,counterSHOCKS,'b','Linewidth',2);
plot(time,(UEq06-UEq06(1))-counter,'r','Linewidth',2); hold on;
plot(time,counter2010Q3,'b--','Linewidth',2);
ylim([-1 6]); 
xlim([2006.75 2013.25]);
ylabel('Unemployment Rates');
legend('UE during 2007 Recession','Predicted UE from Financial Shocks','Unexplained UE');



sdfv

%========================================================================
% sensitivity of GZ to other VAR specifications
clear all
VARlags=8;

GZdata;                % loads quarterly data in GZ from 1973Q1-2010Q3
ebp=GZ(:,1);           % excess bond premium
C=GZ(:,2);             % real consumption
I=GZ(:,3);             % real business investment
Y=GZ(:,4);             % real GDP
P=GZ(:,5);             % GDP imp. price deflator
XS=GZ(:,6);            % excess stock return
TB=GZ(:,7);            % 10-year Treasury rate
FFR=GZ(:,8);           % effective FFR

UE_q;                  % load quarterly UE from 1968Q1-2010Q3.
UE=ue(21:length(ue));

% first difference relevant variables
dC=100*(log(C(2:length(ebp),1))-log(C(1:length(ebp)-1)));
dI=100*(log(I(2:length(ebp),1))-log(I(1:length(ebp)-1)));
dY=100*(log(Y(2:length(ebp),1))-log(Y(1:length(ebp)-1)));
dP=100*(log(P(2:length(ebp),1))-log(P(1:length(ebp)-1)));
ebp=ebp(2:length(ebp)); XS=XS(2:length(XS)); TB=TB(2:length(TB)); FFR=FFR(2:length(FFR));

% make lags for VAR
dC=makelags(dC,VARlags); dI=makelags(dI,VARlags); dY=makelags(dY,VARlags); dP=makelags(dP,VARlags); 
ebp=makelags(ebp,VARlags); TB=makelags(TB,VARlags); XS=makelags(XS,VARlags); FFR=makelags(FFR,VARlags);
%due=UE(2:length(ue))-UE(1:length(ue)-1);
UE=makelags(UE(2:length(UE)),VARlags);
%dUE=makelags(due,VARlags);

% run VARs
options.irfstd=0;
dat_base=[dC(:,1:3) dI(:,1:3) UE(:,1:3) dP(:,1:3) ebp(:,1:3) XS(:,1:3) TB(:,1:3) FFR(:,1:3)];
out_GZ=varcg(dat_base,2,options);     

dat_Lags=[dC dI UE dP ebp XS TB FFR];
out_Lags=varcg(dat_Lags,VARlags,options);     

dat_small=[UE dP ebp XS TB FFR];
out_small=varcg(dat_small,VARlags,options);     

dat_short=dat_base(1:length(dat_base)-7,:);
out_short=varcg(dat_short,2,options);     

figure(4)
plot(1:20,out_GZ.irf(1:20,3,5),'k','Linewidth',2); hold on
plot(1:20,out_Lags.irf(1:20,3,5),'r','Linewidth',2); 
plot(1:20,out_small.irf(1:20,1,3),'b','Linewidth',2); 
plot(1:20,out_short.irf(1:20,3,5),'g','Linewidth',2); 
legend('baseline','more lags','drop C&I, more lags','stop 2008');

%========================================================================
% incorporate RR shocks into GZ VAR in lieu of FFR
clear all
VARlags=2;

GZdata;                % loads quarterly data in GZ from 1973Q1-2010Q3
ebp=GZ(:,1);           % excess bond premium
C=GZ(:,2);             % real consumption
I=GZ(:,3);             % real business investment
Y=GZ(:,4);             % real GDP
P=GZ(:,5);             % GDP imp. price deflator
XS=GZ(:,6);            % excess stock return
TB=GZ(:,7);            % 10-year Treasury rate
FFR=GZ(:,8);           % effective FFR

UE_q;                  % load quarterly UE from 1968Q1-2010Q3.
UE=ue(21:length(ue));

% first difference relevant variables
dC=100*(log(C(2:length(ebp),1))-log(C(1:length(ebp)-1)));
dI=100*(log(I(2:length(ebp),1))-log(I(1:length(ebp)-1)));
dY=100*(log(Y(2:length(ebp),1))-log(Y(1:length(ebp)-1)));
dP=100*(log(P(2:length(ebp),1))-log(P(1:length(ebp)-1)));
ebp=ebp(2:length(ebp)); XS=XS(2:length(XS)); TB=TB(2:length(TB)); FFR=FFR(2:length(FFR));

% make lags for VAR
dC=makelags(dC,VARlags); dI=makelags(dI,VARlags); dY=makelags(dY,VARlags); dP=makelags(dP,VARlags); 
ebp=makelags(ebp,VARlags); TB=makelags(TB,VARlags); XS=makelags(XS,VARlags); FFR=makelags(FFR,VARlags);
UE=makelags(UE(2:length(UE)),VARlags);

% baseline variables for GZ (2012)
dat_base=[dC dI UE dP ebp XS TB FFR];

% Romer Romer (extended sample shocks)
USdata;                     % load UE and rrsh (UE rate and Romer Romer mp shocks updated from 1968:1 to Dec. 2008).
% make RR shocks quarterly
ind=1;
for j=1:length(rrsh)/3
    rrQ(j)=mean(rrsh(ind:ind+2));
    ind=ind+3;
end
rrQ=rrQ(21:length(rrQ));
rrQ=makelags(rrQ,VARlags);

dat_RR=[dat_base(1:length(rrQ),1:7*(VARlags+1)) rrQ];

% run VARs
options.irfstd=1;
out_GZ=varcg(dat_base(1:length(rrQ),:),VARlags,options);     
out_RR=varcg(dat_RR,VARlags,options);     

scale=max(out_GZ.irf(1:20,3,5))/max(out_RR.irf(1:20,3,8));

figure(5)
plot(1:20,out_GZ.irf(1:20,3,5),'k','Linewidth',2); hold on
plot(1:20,scale*out_RR.irf(1:20,3,8),'r','Linewidth',2); 
legend('bond premium','monetary policy');
