%  irf_comp.m

% comparison of IRF of UE to monetary shocks (Romer Romer) and financial
% shocks (Gilchrist Zachrajsek).

clear all
close all

%=========================================================================
%  replicate GZ results and generate IRF of UE
GZdata;                % loads quarterly data in GZ from 1973Q1-2010Q3
ebp=GZ(:,1);           % excess bond premium
C=GZ(:,2);             % real consumption
I=GZ(:,3);             % real business investment
Y=GZ(:,4);             % real GDP
P=GZ(:,5);             % GDP imp. price deflator
XS=GZ(:,6);            % excess stock return
TB=GZ(:,7);            % 10-year Treasury rate
FFR=GZ(:,8);           % effective FFR

UE_q;                  % load quarterly UE from 1968Q1-2010Q3.

% first difference relevant variables
dC=100*(log(C(2:length(ebp),1))-log(C(1:length(ebp)-1)));
dI=100*(log(I(2:length(ebp),1))-log(I(1:length(ebp)-1)));
dY=100*(log(Y(2:length(ebp),1))-log(Y(1:length(ebp)-1)));
dP=100*(log(P(2:length(ebp),1))-log(P(1:length(ebp)-1)));

% make lags for VAR
dC=makelags(dC,2); dI=makelags(dI,2); dY=makelags(dY,2); dP=makelags(dP,2); 
ebp=makelags(ebp(2:length(ebp)),2); TB=makelags(TB(2:length(TB)),2); XS=makelags(XS(2:length(XS)),2); FFR=makelags(FFR(2:length(FFR)),2);
UE=makelags(ue(22:length(ue)),2);
due=ue(2:length(ue))-ue(1:length(ue)-1);
dUE=makelags(due(21:length(due)),2);

% run VAR
dat=[dC dI dY dP ebp XS TB FFR];
options.irfstd=1;
out_GZ=varcg(dat,2,options);     

dat_UE=[dC dI UE dP ebp XS TB FFR];
options.irfstd=1;
out_UE=varcg(dat_UE,2,options);     

dat_dUE=[dC dI dUE dP ebp XS TB FFR];
options.irfstd=1; options.Nirfstd=100;
out_dUE=varcg(dat_dUE,2,options);     

%==========================================================================
% now estimate IRF of UE to Romer and Romer shock (extended sample, monthly)
USdata;                     % load UE and rrsh (UE rate and Romer Romer mp shocks updated to Dec. 2008).
ARlag=24;
MAlag=36;
T=60;
maxlag=max(ARlag,MAlag);
UE=makelags(UE,maxlag);
rrsh=makelags(rrsh,maxlag);
irf=impulse(UE(:,1),[ones(length(UE),1) UE(:,2:ARlag+1) rrsh(:,2:MAlag+1)],T,ARlag+2,1) ;
sd=se(UE(:,1),[ones(length(UE),1) UE(:,2:ARlag+1) rrsh(:,2:MAlag+1)],T,ARlag+2,100,1) ;
irf_UB=irf+sd;
irf_LB=irf-sd;

% make RR responses quarterly
ind=1;
for j=1:20
    irfRR(j)=mean(irf(ind:ind+2));
    irfUB(j)=mean(irf_UB(ind:ind+2));
    irfLB(j)=mean(irf_LB(ind:ind+2));
    ind=ind+3;
end

% estimate GZ stopping in Dec. 2008
% scaling factor so IRF of UE is same scale
scale=max(out_UE.irf(1:20,3,5))/max(irfRR);


%=========================================================================
% now estimate IRF of UE to Romer and Romer shock (ORIGINAL sample, monthly)
MPshockdata;                        % load macro data: IP, UE, CPI, CPIc, PPI, FFR, PCOM: 1968:1-1996:12, monthly
RRshock;                            % load MP shock measure from Romer and Romer (2004): uRR 1969:1-1996:12
UE2=makelags(UE,60);
uRR2=makelags([zeros(12,1); uRR],60);

irf=impulse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,1) ;
sd=se(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,100,1) ;
irf_UB=irf+1.64*sd;
irf_LB=irf-1.64*sd;

% make RR responses quarterly
ind=1;
for j=1:20
    irfRR(j)=mean(irf(ind:ind+2));
    irfUB(j)=mean(irf_UB(ind:ind+2));
    irfLB(j)=mean(irf_LB(ind:ind+2));
    ind=ind+3;
end

% scaling factor so IRF of UE is same scale
scale=max(out_UE.irf(1:20,3,5))/max(irfRR);

%=======================================================================
%  construct estimates of UE responses "after peak" and half-lives
%  for both RR 2004 and GZ 2012.
N=2000;                              % number of simulations
MPshockdata;                        % load macro data: IP, UE, CPI, CPIc, PPI, FFR, PCOM: 1968:1-1996:12, monthly
RRshock;                            % load MP shock measure from Romer and Romer (2004): uRR 1969:1-1996:12
UE2=makelags(UE,60);
uRR2=makelags([zeros(12,1); uRR],60);

% Romer and Romer monthly responses after peak UE
irf=peakresponse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,1) ;   
dist=dist_peakresponse(UE2(:,1),[ones(length(uRR2),1) UE2(:,2:25) uRR2(:,2:37)],60,26,N+1,1) ;
dist_sort=sort(dist);

% make quarterly
dist_q=ones(rows(dist),1);
for j=2:18
    dist_q(:,j)=mean(dist_sort(:,3*(j-2)+2:3*(j-2)+4)')';
end

% GZ quarterly responses after peak UE
%dat_UE=[dC dI UE dP ebp XS TB FFR];
options.Nirfstd=N; options.var=3; options.shock=5;
out_gz1=varcg_peak(dat_UE,2,options);     
dist_gz=out_gz1.impdist;
dist_gz=sort(dist_gz);
dist_gz=dist_gz(:,1:cols(dist_q));

% plot the two distributions of UE responses after peak.
figure(81)
%subplot(1,2,1)
plot(0:cols(dist_q)-1,dist_q(floor(0.5*rows(dist)),:),'k','Linewidth',2); hold on
plot(0:cols(dist_q)-1,dist_q(floor(0.95*rows(dist)),:),'k--','Linewidth',2);
plot(0:cols(dist_q)-1,dist_gz(floor(0.5*rows(dist)),:),'bo-','Linewidth',2,'MarkerFaceColor','white'); 
xv = [(0:cols(dist_q)-1),fliplr((0:cols(dist_q)-1))];
yv = [dist_gz(floor(0.95*rows(dist)),:),fliplr(dist_gz(floor(0.05*rows(dist)),:))];
hReg = fill(xv,yv,[0.85 0.85 0.85],'EdgeColor','none'); hold on
plot(0:cols(dist_q)-1,dist_q(floor(0.5*rows(dist)),:),'k','Linewidth',2); hold on
plot(0:cols(dist_q)-1,dist_gz(floor(0.5*rows(dist)),:),'bo-','Linewidth',2,'MarkerFaceColor','white'); 
plot(0:cols(dist_q)-1,dist_q(floor(0.95*rows(dist)),:),'k--','Linewidth',2);
legend('UE Persistence after Monetary Policy Shock','UE Persistence after Monetary Policy Shock (90% CI)','UE Persistence after Financial Shock','UE Persistence after Financial Shock (90% CI)');
plot(0:cols(dist_q)-1,dist_q(floor(0.05*rows(dist)),:),'k--','Linewidth',2);
%plot(0:cols(dist_q)-1,dist_gz(floor(0.95*rows(dist)),:),'b:','Linewidth',2);
%plot(0:cols(dist_q)-1,dist_gz(floor(0.05*rows(dist)),:),'b:','Linewidth',2);
plot(0:cols(dist_q)-1,zeros(1,cols(dist_q)),'k','Linewidth',1);
xlabel('Quarters after Peak Effect of Shock on Unemployment');
ylabel('Unemployment relative to Peak Rise');
xlim([0 12]);
ylim([-.8 1]);

% plot equivalent dynamics for UE gaps after typical recession and Great Recession
UE_persistencedata;
figure(82)
%subplot(1,2,2)
plot(0:12,Ppre90,'ro-','Linewidth',2,'MarkerFaceColor','white'); hold on
plot(0:12,P2007,'k','Linewidth',2); 
xv = [(0:12),fliplr(0:12)];
yv = [dist_gz(floor(0.95*rows(dist)),1:13),fliplr(dist_gz(floor(0.05*rows(dist)),1:13))];
hReg = fill(xv,yv,[0.85 0.85 0.85],'EdgeColor','none'); hold on
plot(0:12,Ppre90,'ro-','Linewidth',2,'MarkerFaceColor','white'); hold on
plot(0:12,P2007,'k','Linewidth',2); 
%plot(0:12,dist_gz(floor(0.95*rows(dist)),1:13),'k:','Linewidth',2);
%plot(0:12,dist_gz(floor(0.05*rows(dist)),1:13),'k:','Linewidth',2);
plot(0:12,zeros(1,13),'k','Linewidth',1);
legend('UE Persistence in Average Pre-1990 Recessions','UE Persistence in 2007 Recession','UE Persistence after Financial Shock (90% CI)');
xlabel('Quarters after Peak Effect of Shock on Unemployment');
ylabel('Unemployment relative to Peak Rise');
ylim([-.8,1]);
xlim([0 12]);


%=========================================================================
%  now construct rise in UE predicted by financial shocks starting in 2007.

options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=134;                          % start shocks in 2007Q1
options.N=15;                               % feed in shocks until 2010Q3
options.T=26;                               % counterfactual until 2013Q2
out=varcg_counter(dat_UE,2,options);        % this constructs counterfactual
counter=[0 ; out.counter];
counter2010Q3=[nan(15,1); counter(16:length(counter))]; % counterfactuals starting 2010Q3
counterSHOCKS=[counter(1:16); nan(11,1)];

UE_2006q4;                                  % load quarterly UE (UEq06): 2006Q4-2013Q2, plus recession07 dummy variable (-100,100)
time=(2006.75:0.25:2013.25);


scale=max(UEq06-UEq06(1));
figure(74)
subplot(1,2,2)
plot(time,(UEq06-UEq06(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/scale,'b','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counterSHOCKS)/scale,'r','Linewidth',1.5); hold on;
plot(time,counter2010Q3/scale,'b--','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counter)/scale,'r--','Linewidth',1.5); hold on;
%area(time68,recession68*100,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
area(time,recession07,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq06-UEq06(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counterSHOCKS/scale,'b','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counterSHOCKS)/scale,'r','Linewidth',1.5); hold on;
plot(time,counter2010Q3/scale,'b--','Linewidth',4);
plot(time,((UEq06-UEq06(1))-counter)/scale,'r--','Linewidth',1.5); hold on;
ylim([-.5 1.5]); 
xlim([2006.75 2013.25]);
ylabel('Unemployment Rate');
legend('Actual (Normalized) UE','Predicted UE from Financial Shocks','Unexplained UE');

%========================================================================
% construct counterfactual UE paths from financial shocks for 1990, 2001
% and 2007 recessions.
%========================================================================

% 1990 recession
options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=65;                           % start shocks in 1989Q4
options.N=15;                               % feed in shocks through 1993Q4
options.T=26;                               % counterfactual until 1996Q1
out_90=varcg_counter(dat_UE,2,options);     % this constructs counterfactual
counter90=[0 ; out_90.counter];

UE_1989q3;                                  % load quarterly UE (UEq89): 1989Q3-1996Q1, plus recession90 dummy variable (-100,100)
time=(1989.5:0.25:1996);
scale=max(UEq89-UEq89(1));

% 2001 recession
options.var=3;                              % get counterfactual UE
options.shock=5;                            % get counterfactual after financial shock
options.start=107;                          % start shocks in 2000Q2
options.N=15;                               % feed in shocks through 2003Q4
options.T=26;                               % counterfactual until 2006Q3
out_01=varcg_counter(dat_UE,2,options);     % this constructs counterfactual
counter01=[0 ; out_01.counter];

UE_2000q1;                                  % load quarterly UE (UEq00): 2000Q1-2006Q3, plus recession01 dummy variable (-100,100)
time=(2000:0.25:2006.5);
scale=max(UEq00-UEq00(1));
figure(74)
subplot(1,2,1)
plot(time,(UEq00-UEq00(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counter01/scale,'b','Linewidth',2);
plot(time,((UEq00-UEq00(1))-counter01)/scale,'r','Linewidth',1.5); hold on;
area(time,recession01,-100,'FaceColor',[.9 0.9 0.9],'EdgeColor','none'); hold on;
plot(time,(UEq00-UEq00(1))/scale,'ko-','Linewidth',2,'MarkerFaceColor','white'); hold on;
plot(time,counter01/scale,'b','Linewidth',4);
plot(time,((UEq00-UEq00(1))-counter01)/scale,'r','Linewidth',1.5); hold on;
ylim([-.5 1.5]); 
xlim([2000 2006.5]);
ylabel('Unemployment Rate');
