clear all
set more off
#delimit;

/*************************************************************************************************************
This do-file calls programs that perform the event studies. It then makes the table for publication.

*************************************************************************************************************/
do Paths;
tempfile data;
local Stock_id symbol;
local CDS_id ticker;
local Bond_id company_symbol;
local CDX_id Median;
local Stock_depvar Dlogprice;
local CDS_depvar Dintradayspread;
local Bond_depvar Dyld_pt;
local CDX_depvar Dlogintraday100;
local CDX_sector IG;
local LI_name `"Insurance company"';
local BHC_name `"Bank"';
local lower 2;
local upper 18;
local alldates alldates;

/*************************************************************************************************************
1. Run programs
*************************************************************************************************************/

foreach sector in LI BHC `CDX_sector' SP500 {;

	tempfile `sector'_data;
	local `sector'_vars0 `"DytmBP0"';
	local `sector'_vars1 `"DytmBP1"';
	local `sector'_nmeasures = 0;

	foreach measure in CDS Bond Stock CDX {;
	
		capture estimates drop _all;
	
		if (`"`measure'"'==`"CDX"' & !inlist(`"`sector'"',`"IG"',`"HY"')) | (`"`measure'"' == `"Bond"' & !inlist(`"`sector'"',`"LI"',`"BHC"')) | (`"`measure'"'==`"Stock"' & inlist(`"`sector'"',`"IG"',`"HY"')) | (`"`measure'"'==`"CDS"' & inlist(`"`sector'"',`"SP500"',`"IG"',`"HY"')) {;
			continue;
		};
		
		di `"`sector' `measure'"';
		
		if inlist(`"`sector'"',`"LI"',`"BHC"') {;
			local regressionfiles `"`regressionfiles' `sector'-`measure'-response"';
		};
		
		local `sector'_nmeasures = ``sector'_nmeasures' + 1; 
		foreach stat in Median Mean vwMean {;
			local `sector'_`stat'_vars0 `"``sector'_`stat'_vars0' `measure'`stat'0"';
			local `sector'_`stat'_vars0 `"``sector'_`stat'_vars1' `measure'`stat'1"';
			local `sector'_`stat'_vars `"``sector'_`stat'_vars' `measure'`stat'`sector'"';
		};
		if inlist(`"`sector'"',`"LI"',`"BHC"') & `"`measure'"'==`"Stock"' {;
			local `sector'_headers `"``sector'_headers' & \multicolumn{1}{L}{`measure'$^\text{d}$}"';
		};
		else if inlist(`"`sector'"',`"LI"',`"BHC"') & `"`measure'"'==`"CDS"' {;
			local `sector'_headers `"``sector'_headers' & \multicolumn{1}{L}{`measure'$^\text{b}$}"';
		};			
		else if inlist(`"`sector'"',`"LI"',`"BHC"') & `"`measure'"'==`"Bond"' {;
			local `sector'_headers `"``sector'_headers' & \multicolumn{1}{L}{`measure'$^\text{c}$}"';
		};	
		else if inlist(`"`sector'"',`"SP500"') & `"`measure'"'==`"Stock"' {;
			local `sector'_headers `"``sector'_headers' & \multicolumn{1}{L}{`measure'$^\text{f}$}"';
		};	
		else if inlist(`"`sector'"',`"`CDX_sector'"') & `"`measure'"'==`"CDX"' {;
			local `sector'_headers `"``sector'_headers' & \multicolumn{1}{L}{`measure'$^\text{e}$}"';
		};	
	
		/*Run file that executes event study*/
		if `"`measure'"'==`"Bond"' {;
			*EventStudy`measure' `sector', lower(`lower') upper(`upper') yield;
		};
		else {;
			*EventStudy`measure' `sector', lower(`lower') upper(`upper');
		};

		/*Merge with other measures*/
		if `"`alldates'"'==`""' | `"`sector'"'!=`"LI"' {;
			qui use ``measure'_id' date DytmBP ``measure'_depvar' Pvalue using "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'" if inlist(``measure'_id',`"Median"',`"Mean"',`"vwMean"'), clear;	
		};
		else if `"`alldates'"'==`"alldates"' & `"`sector'"'==`"LI"' {;
			qui use ``measure'_id' date DytmBP ``measure'_depvar' Pvalue using "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'alldates" if inlist(``measure'_id',`"Median"',`"Mean"',`"vwMean"'), clear;	
		};
		rename ``measure'_depvar' `measure';
		if `"`measure'"' == `"CDX"' {;
			rename `measure' CDSMedian;
			qui gen CDSMean = CDSMedian;
			qui gen CDSvwMean = CDSMedian;
			qui gen CDSMeanPvalue = .;
			qui gen CDSvwMeanPvalue = .;
			qui drop Median;
		};
		else {;
			qui reshape wide `measure' Pvalue, i(date DytmBP) j(``measure'_id') string;
			qui drop PvalueMedian;
			rename PvalueMean `measure'MeanPvalue;
			rename PvaluevwMean `measure'vwMeanPvalue;
		};
		qui gen sector = `"`sector'"';
		foreach stat in Median Mean vwMean MeanPvalue vwMeanPvalue {;
			local `sector'_`stat'_measures `"``sector'_`stat'_measures' `measure'`stat'"';
			local `sector'_measures `"``sector'_measures' `measure'`stat'"';
		};
		if `"``sector'_First'"'==`""' {;
			qui save ``sector'_data';
			local `sector'_First `"No"';
		};
		else {;
			merge 1:1 date using ``sector'_data', nogenerate;
			qui save ``sector'_data', replace;
		};
	};
	
	if `"`First'"'==`""' {;
		qui save `data';
		local First `"No"';
	};
	else {;
		qui append using `data';
		qui save `data', replace;
	}; 
	
	local `sector'_headers2 `"``sector'_headers'"';
	local `sector'_headers `"\multicolumn{1}{c}{T-note$^\text{a}$} ``sector'_headers'"';

};

if `"`alldates'"'!=`"alldates"' {;
	qui drop if inlist(date,td(18dec2013));
};
/*************************************************************************************************************
2a. Paper table of changes on each date
*************************************************************************************************************/
qui replace sector = `"Market"' if inlist(sector,`"SP500"',`"IG"',`"HY"');
qui collapse (mean) `LI_measures' DytmBP, by(date sector);
qui reshape wide `LI_measures' DytmBP, i(date) j(sector) string;

qui format date %tdnn/dd/YY;
foreach var of varlist Stock* Bond* {;
	if regexm(`"`var'"',`"Pvalue"') {;
		continue;
	};
	qui replace `var' = `var' * 100;
};

qui gen date_string = string(date,`"%tdnn/dd/YY"');
qui replace date_string = `"Initial QE$^\text{g}$"' if date==1;
qui replace date_string = `"Taper$^\text{h}$"' if date==2;
qui replace date_string = `"Sample end$^\text{i}$"' if date==3;

qui replace date_string = `"\rowcolor{light-gray} "' + date_string if DytmBPLI>0; /*Colors contractionary changes in grey*/

qui set obs `=`=_N'+1';
qui gen order = _n;
qui replace order = order + `=_N' if inrange(date,1,3);
qui sort order;

local filenames;
local Median_name Median;
local Mean_name Mean;
local vwMean_name `"Value-weighted mean"';
foreach stat in Median Mean vwMean {;
	qui format DytmBPLI `LI_`stat'_vars' `BHC_`stat'_vars' Stock`stat'Market CDS`stat'Market %9.1f;
	capture qui replace Stock`stat'PvalueMarket = . if regexm(date_string,`"Sample end"');
	/*Add significance stars if mean or value-weighted mean and format numbers*/
	if inlist(`"`stat'"',`"Mean"',`"vwMean"') {;
		foreach var in `LI_`stat'_vars' `BHC_`stat'_vars' Stock`stat'Market {;
			local Pvalue = regexr(`"`var'"',`"`stat'"',`"`stat'Pvalue"');
			qui gen `var'_string = string(round(`var',0.1));
			qui replace `var'_string = `""' if `var'_string==`"."';
			qui replace `var'_string = `"-0"' + string(abs(round(`var',0.1))) if inrange(`var',-0.94999,-0.05);
			qui replace `var'_string = `var'_string + `".0"' if round(`var',0.1)==round(`var',1) & !missing(`var');
			qui replace `var'_string = `var'_string + `"\$^{+}$"' if inrange(`Pvalue',0.05,0.1);
			qui replace `var'_string = `var'_string + `"\$^{*}$"' if inrange(`Pvalue',0.01,0.05);
			qui replace `var'_string = `var'_string + `"\$^{**}$"' if `Pvalue'<0.01;
			if `"`var'"'!=`"Stock`stat'Market"' {;
				local `stat'_vars `"``stat'_vars' `var'_string"';
			};
			else {;
				local `stat'_vars `"``stat'_vars' CDS`stat'Market `var'_string"';
			};
		};
		local signote `" +,*,** indicate significance at the 0.1, 0.05, 0.01 levels, respectively, based on the larger of the conventional or robust standard error from a regression of the change in the asset price on a constant on the date indicated."';
	};
	else {;
		local `stat'_vars `"`LI_`stat'_vars' `BHC_`stat'_vars' CDS`stat'Market Stock`stat'Market"';
	};
	local filename event-study-by-date-`stat'-window-`lower'-`upper';
	local nmeasures = `LI_nmeasures' + `BHC_nmeasures' + `SP500_nmeasures' + ``CDX_sector'_nmeasures';
	local notes `" 
	Notes:`signote'\\ 
	a. Change in the on-the-run five year Treasury note during announcement window covering two minutes before to eighteen minutes after announcement, basis points.\\
	b. ``stat'_name' change in the CDS spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), 5 year tenor, basis points.\\
	c. ``stat'_name' change in the bond yield during announcement window covering at least two minutes before to eighteen minutes after announcement, basis points.\\
	d. ``stat'_name' change in the log stock price during announcement window covering two minutes before to eighteen minutes after announcement, log points.\\
	e. Log change in the on-the-run North American `CDX_sector' CDX index, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm). Significance thresholds not computed.\\
	f. ``stat'_name' change in the log stock price of companies in the S\&P 500, excluding life insurers and bank holding companies, during announcement window covering two minutes before to eighteen minutes after announcement, log points.\\
	g. 12/16/2008 and 3/18/2009. Totals may differ due to rounding or sample composition.\\
	h. 5/22/2013 and 6/19/2013. Totals may differ due to rounding or sample composition.\\
	i. 7/10/2013 and 9/18/2013. Totals may differ due to rounding or sample composition.
	"'; 
	local title `"Event study effects of unconditional monetary policy"';
	local headlines1 `""&\multicolumn{1}{L}{Treasury$^\text{a}$}&\multicolumn{`LI_nmeasures'}{c}{Life insurers}& \multicolumn{`BHC_nmeasures'}{c}{Bank holding companies}& \multicolumn{2}{c}{Market}\\ \cmidrule(l{.75em}){3-`=3+`LI_nmeasures'-1'} \cmidrule(l{.75em}){`=3+`LI_nmeasures''-`=3+`LI_nmeasures'+`BHC_nmeasures'-1'} \cmidrule(lr{.75em}){`=3+`LI_nmeasures'+`BHC_nmeasures''-`=3+`LI_nmeasures'+`BHC_nmeasures'+1'}" "';
	local headlines2 `"" &  `LI_headers2'  `BHC_headers2'  ``CDX_sector'_headers2' `SP500_headers2' \\ ""';
	local si_fmt `"2.2"';
	listtex date_string DytmBPLI ``stat'_vars' using `"$OutputPath/`filename'.tex"', 
	replace rstyle(tabular) headline("\begin{table}[!tp]\centering" "\sisetup{table-format=`si_fmt'}" "\def\sym X 1{\ifmmode^{ X 1}\else\(^{ X 1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\textwidth}{l *{`=`nmeasures'+1'}{S}}" "\\" "\hline \hline" `headlines1' `headlines2' "\hline") footline("\hline \hline" "\multicolumn{`=`nmeasures'+2'}{l}{" "\begin{minipage}{\hsize}" "\rule{0pt}{9pt}" "\footnotesize" "\noindent" "`notes'"  "\end{minipage}" "}\\" "\end{tabularx}" "\label{tab:event-study-by-date}" "\end{table}"); 
	local filenames `"`filenames' `filename'.tex"';
};
EmbedTex `"`filenames'"' using `"$OutputPath/Event-studies-by-date-window-`lower'-`upper'`alldates'.tex"', title(Event Studies) author(Gabriel Chodorow-Reich) replace;
if `lower'==2 & `upper'==18 & `"`alldates'"'==`""' {;
	qui copy `"$OutputPath/event-study-by-date-vwMean-window-`lower'-`upper'.tex"' `"$PaperPath/event-study-by-date.tex"', replace;
};x;
/*************************************************************************************************************
2b. Presentation table of summary changes
*************************************************************************************************************/
local stat vwMean;
local notes;
local nmeasures = 5;
local LI_desc `"Life insurers"';
local BHC_desc `"Bank holding companies"';

foreach sector in LI BHC {;
	local filename event-study-by-date-`sector'-`stat'-window-`lower'-`upper';
	local title `"``sector'_name' summary effects"';
	local headlines1 `""&\multicolumn{1}{L}{Treasury}&\multicolumn{``sector'_nmeasures'}{c}{``sector'_desc'}& \multicolumn{1}{L}{N.A.}& \multicolumn{1}{L}{Non-}\\ " "';
	local headlines2 `""&&\multicolumn{``sector'_nmeasures'}{c}{value-weighted}& \multicolumn{1}{L}{IG}& \multicolumn{1}{L}{finance}\\ \cmidrule(l{.75em}){3-`=3+``sector'_nmeasures'-1'}" "';
	local headlines3 `" &  ``sector'_headers2' & \multicolumn{1}{L}{CDX} & \multicolumn{1}{L}{S\&P 500}\\ "';
	tokenize `headlines3';
	while `"`1'"'!=`""' {; 
		local newheader `" `=regexr(`"`1'"',`"\$\^\\text{[abcdefghi]}\$"',`""')' "'; /*Removes footnote headers*/
		local headlines3 = subinstr(`"`headlines3'"',`"`1'"',`"`newheader'"',1); 
		macro shift;
	};
	local headlines3 `""`headlines3'""';
	qui replace date_string = regexr(date_string,`"\\$\^\\text{[abcdefghi]}\\$"',`""') if inrange(date,1,3); /*Removes footnote headers*/
	qui replace date_string = regexr(date_string,`"\\rowcolor{light-gray}"',`""') if inrange(date,1,3); /*Removes footnote headers*/

	preserve;
	qui keep if inrange(date,1,3);
	qui replace date_string = "Summer 2013" if regexm(date_string,"Sample end");
	qui replace date_string = "Taper 2013" if regexm(date_string,"Taper");
	qui gen p_order = _n;
	qui set obs `=`=_N'+4';
	qui replace p_order = p_order[_n-3]-0.5 if missing(p_order);
	qui replace p_order = 3.5 in `=_N';
	qui sort p_order;
	
	listtex date_string DytmBPLI CDS`stat'`sector'_string Bond`stat'`sector'_string Stock`stat'`sector'_string CDS`stat'Market Stock`stat'Market_string using `"$PresentationPath/`filename'.tex"', 
	replace rstyle(tabular) headline("\begin{table}[!tp]\centering" "\sisetup{table-format=`si_fmt'}" "\def\sym X 1{\ifmmode^{ X 1}\else\(^{ X 1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\textwidth}{l *{`=`nmeasures'+1'}{S}}" "\\" "\hline \hline" "\\" `headlines1' `headlines2' `headlines3' "\hline") footline("\hline \hline" "\multicolumn{`=`nmeasures'+2'}{l}{" "\begin{minipage}{\hsize}" "\rule{0pt}{9pt}" "\footnotesize" "" "\noindent" "`notes'"  "\end{minipage}" "}\\" "\end{tabularx}" "\label{tab:event-study-by-date}" "\end{table}"); 
	restore;

};

/*Drop combined dates*/
qui drop if inrange(date,1,3);

/*************************************************************************************************************
3. Regressions of change, separate for full sample and excluding winter 2008-09
*************************************************************************************************************/
qui gen DytmBP10 = DytmBPLI*0.1;
label variable DytmBP10 `"LatexMath \Delta$ Treasury (10 b.p.)"';
local filenames;
foreach stat in Median Mean vwMean {;
	local n_total;
	foreach sector in LI BHC {;
		local n_`sector'; local group`sector'; local midrule`sector';
		foreach measure in CDS Bond Stock {;
			foreach condition in `""' `"if date>td(18mar2009)"' {;
				local n_`sector' = `n_`sector'' + 1;
				local n_total = `n_total' + 1;
				qui eststo: reg `measure'`stat'`sector' DytmBP10 `condition';
				if strmatch(`"`condition'"',`"if *"') {;
					qui estadd local Winter08 `"No"';
				};
				else {;
					qui estadd local Winter08 `"Yes"';
				};	
			};
			local group`sector' `"`group`sector'' &\multicolumn{2}{c}{`measure'}"';
			local midrule`sector' `"`midrule`sector'' \cmidrule(lr{.75em}){`n_total'-`=`n_total'+1'}"';
		};
	};
	local groups1 `"& \multicolumn{`n_LI'}{c}{Life insurers} & \multicolumn{`n_BHC'}{c}{Bank holding companies}"'; 
	local midrules1 `"\cmidrule(l{.75em}){2-`=`n_LI'+1'} \cmidrule(lr{.75em}){`=`n_LI'+2'-`=`n_total'+1'}"';
	local groups2 `"& \multicolumn{`n_total'}{c}{Dependent variable: change in}"'; 
	local groups3 `"`groupLI' `groupBHC'"';
	local midrules3 `"`midruleLI' `midruleBHC'"';
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\ `midrules2'" "`groups3'\\ `midrules3'" "';	

	local stats "Winter08 r2 N";
	local stats_fmt "%3s %9.2f %9.0f";
	local stats_label `" `"Includes winter 2008-09"' `"LatexMath R^2$"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 2.3;
	local title "``stat'_name' price response regressions";
	local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n_total'}{S}}" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n_total'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
	local sectype 5;
	local notes `"Notes: The dependent variable is the `=lower(`"``stat'_name'"')' change in the log stock price during announcement window covering two minutes before to eighteen minutes after announcement, in log points, the median change in the CDS spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), 5 year tenor, in basis points, or the median change in the log bond price during announcement window covering at least two minutes before to eighteen minutes after announcement, in log points, as indicated by the table header. The variable LatexMath \Delta$ Treasury (10 b.p.) is the change in the yield to maturity of the on-the-run `sectype' year Treasury from `lower' minutes before to `upper' minutes after the announcement, in 10 basis point increments. Standard errors in parentheses. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`stat'-price-response-window-`lower'-`upper'`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'.tex"';
	esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(%9.2f)) se(par fmt(%9.2f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{1}{c}" "{1}{L}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
	estimates drop _all;
};

EmbedTex `"`filenames'"' using `"$OutputPath/Event-studies-by-date-window-`lower'-`upper'.tex"', title(Event Studies) author(Gabriel Chodorow-Reich) landscape append;
*qui copy `"$OutputPath/Median-price-response.tex"' `"$PaperPath/Price-response"', replace;

/*************************************************************************************************************
4. Regressions of change, interacting Treasury change with dummy for winter 2008-09
*************************************************************************************************************/
qui gen DytmBP10Xwinter = DytmBP10 * (date<=td(21mar2009));
label variable DytmBP10Xwinter `"10 b.p. LatexMath \Delta$ Treasury X winter 2008-09"';
qui gen DytmBP10XNotwinter = DytmBP10 * (date>td(21mar2009));
label variable DytmBP10XNotwinter `"10 b.p. LatexMath \Delta$ Treasury X post winter 2008-09"';
local filenames;
foreach stat in Median Mean vwMean {;
	local n_total;
	foreach sector in LI BHC {;
		local n_`sector'; local group`sector'; local midrule`sector';
		foreach measure in CDS Bond Stock {;
			local n_`sector' = `n_`sector'' + 1;
			local n_total = `n_total' + 1;
			qui eststo: reg `measure'`stat'`sector' DytmBP10Xwinter DytmBP10XNotwinter;
			local group`sector' `"`group`sector'' &\multicolumn{1}{L}{`measure'}"';
			qui test _b[DytmBP10Xwinter]=_b[DytmBP10XNotwinter];
			qui estadd sca equality = r(p);
		};
	};
	local groups1 `"& \multicolumn{`n_LI'}{c}{Life insurers} & \multicolumn{`n_BHC'}{c}{Bank holding companies}"'; 
	local midrules1 `"\cmidrule(l{.75em}){2-`=`n_LI'+1'} \cmidrule(lr{.75em}){`=`n_LI'+2'-`=`n_total'+1'}"';
	local groups2 `"& \multicolumn{`n_total'}{c}{Dependent variable: change in}"'; 
	local midrules2 `"\cmidrule(lr{.75em}){2-`=`n_total'+1'}"';
	local groups3 `"`groupLI' `groupBHC'"';
	local midrules3 `"\cmidrule(lr{.75em}){2-2} \cmidrule(lr{.75em}){3-3} \cmidrule(lr{.75em}){4-4} \cmidrule(lr{.75em}){5-5} \cmidrule(lr{.75em}){6-6} \cmidrule(lr{.75em}){7-7}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\ `midrules2'" "`groups3'\\ `midrules3'" "';	

	local stats "equality r2 N";
	local stats_fmt "%9.2f %9.2f %9.0f";
	local stats_label `" `"P value of coefficient equality"' `"LatexMath R^2$"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 2.3;
	local title "``stat'_name' price response regressions";
	local table_preamble `" "\begin{table}[tb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n_total'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n_total'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
	local sectype 5;
	local notes `"Notes: The dependent variable is the `=lower(`"``stat'_name'"')' change in the log stock price during announcement window covering two minutes before to eighteen minutes after announcement, in log points, the `=lower(`"``stat'_name'"')' change in the CDS spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), 5 year tenor, in basis points, or the `=lower(`"``stat'_name'"')' change in the log bond price during announcement window covering at least two minutes before to eighteen minutes after announcement, in log points, as indicated by the table header. The variable 10 b.p. LatexMath \Delta$ Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury from `lower' minutes before to `upper' minutes after the announcement, in 10 basis point increments. Winter 2008-09 includes all announcements in December 2008-March 2009. Standard errors in parentheses. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`stat'-price-response-window-`lower'-`upper'-2`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'.tex"';
	esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(%9.2f)) se(par fmt(%9.2f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize \noindent `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{1}{c}" "{1}{L}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
	estimates drop _all;
};

EmbedTex `"`filenames'"' using `"$OutputPath/Event-studies-by-date-window-`lower'-`upper'.tex"', title(Event Studies) author(Gabriel Chodorow-Reich) append;
qui copy `"$OutputPath/vwMean-price-response-window-2-18-2.tex"' `"$PaperPath/Price-response.tex"', replace;

/*************************************************************************************************************
5. Regressions of change, interacting with 2008 stock performance
*************************************************************************************************************/
local n_total; 
foreach sector in LI BHC {;
	local n_`sector'; local group`sector'; local midrule`sector';
	foreach measure in CDS Bond Stock {; 
		preserve;
		qui use "$DataPath/`sector'-`measure'-eventtime" if !inlist(``measure'_id',`"Median"',`"Mean"',`"vwMean"'), clear;
		if inlist(`"`measure'"',`"Bond"',`"Stock"') {;
			qui replace ``measure'_depvar' = 100*``measure'_depvar';
		};
		if `"`measure'"'==`"Bond"' {;
			qui keep if `sector';
		};
		qui egen tag = tag(``measure'_id') if !missing(``measure'_depvar');
		qui sum Dlogprice_2jan2008_24nov2008 if tag;
		qui gen lagged_performance = Dlogprice_2jan2008_24nov2008/r(sd); /*Normalize to have unit variance*/
		qui gen winter0809 = date<=td(21mar2009);
		qui eststo: reg ``measure'_depvar' ibn.date ibn.date#c.lagged_performance, noconstant cluster(``measure'_id');
		local n_total = `n_total' + 1;
		local n_`sector' = `n_`sector'' + 1;
		local group`sector' `"`group`sector'' &\multicolumn{1}{L}{`measure'}"';
		qui estadd local dateFE `"Yes"';
		foreach bname in `: colnames e(b)' {;
			if regexm(`"`bname'"',`"[1-9][0-9][0-9][0-9][0-9][b]?[n]?\.date#c.lagged_performance"') {; 
				local date = string(`=regexr(`"`bname'"',`"[b]?[n]?\..*$"',`""')',`"%tdnn/dd/YY"'); 
				if `"`date'"'==`"12/1/08"' {; 
					local bname = regexr(`"`bname'"',`"[b]?[n]?\."',`"."'); 
				};
				local keepvars `"`keepvars' `bname'"';
				local varlabels `"`varlabels' `bname' `"LatexMath \Delta$ log price pre UMP X `date'"' "';
			};
		};
		restore;
	};
};
		
local groups1 `"& \multicolumn{`n_LI'}{c}{Life insurers} & \multicolumn{`n_BHC'}{c}{Bank holding companies}"'; 
local midrules1 `"\cmidrule(l{.75em}){2-`=`n_LI'+1'} \cmidrule(lr{.75em}){`=`n_LI'+2'-`=`n_total'+1'}"';
local groups2 `"& \multicolumn{`n_total'}{c}{Dependent variable: change in}"'; 
local groups3 `"`groupLI' `groupBHC'"';
local groups `" "`groups1'\\ `midrules1'" "`groups2'\\ `midrules2'" "`groups3'\\" "';	

local stats "dateFE r2 N_clust N";
local stats_fmt "%9.3s %9.2f %9.0f %9.0f";
local stats_label `" `"Date fixed effects"' `"LatexMath R^2$"' `"Firm clusters"' `"Observations"' "';
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 2.3;
local title "Heterogeneous price response regressions";
local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n_total'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n_total'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
local sectype 5;
local notes `"Notes: The dependent variable is the `=lower(`"``stat'_name'"')' change in the log stock price during announcement window covering two minutes before to eighteen minutes after announcement, in log points, the `=lower(`"``stat'_name'"')' change in the CDS spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), 5 year tenor, in basis points, or the `=lower(`"``stat'_name'"')' change in the log bond price during announcement window covering at least two minutes before to eighteen minutes after announcement, in log points, as indicated by the table header. The log stock return pre-UMP is the log stock return from 1/2/08 to 11/24/08, normalized to have unit variance across firms. Standard errors in parentheses. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
local filename `"heterogeneous-price-response-window-`lower'-`upper'`OUTPUT_PS'"';
esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(%9.2f)) se(par fmt(%9.2f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) keep(`keepvars', relax) drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{1}{c}" "{1}{L}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
estimates drop _all;

EmbedTex `"`filename'.tex"' using `"$OutputPath/Event-studies-by-date-window-`lower'-`upper'.tex"', title(Event Studies) author(Gabriel Chodorow-Reich) append;
*qui copy `"$OutputPath/vwMean-price-response2.tex"' `"$PaperPath/Price-response.tex"', replace;

x;

/*************************************************************************************************************
6. Make data set of median changes
*************************************************************************************************************/
qui use `data', clear;
	
/*Assign dates to QE round*/
qui gen QE = "";
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		qui replace QE = `"QE`q'_`i'"' if date==dofC(`=tC(${QE`q'_`i'})');
		local datelist `"`datelist' QE`q'_`i'"';
		local i = `i'+1;
	};
};
qui replace QE = substr(QE,1,3);
qui gen date2 = date;
	
/*Separate by expansionary or contractionary*/
qui gen contractionary = DytmBP>0;
	
/*Collapse by QE round-expansionary*/
qui collapse (min) date (max) date2 (sum) `LI_measures' DytmBP, by(sector QE contractionary);
foreach var of varlist `LI_measures' {;
	qui replace `var' = . if `var'==0;
};
qui replace sector = `"Market"' if inlist(sector,`"SP500"',`"IG"',`"HY"');
qui collapse (min) date (max) date2 (mean) `LI_measures' DytmBP, by(sector QE contractionary);

/*Add date range*/
qui levelsof QE, local(rounds);
foreach round of local rounds {;
	qui sum date if QE==`"`round'"';
	qui replace date = r(min) if QE==`"`round'"';
	qui sum date2 if QE==`"`round'"';
	qui replace date2 = r(max) if QE==`"`round'"';
};			
qui gen daterange = string(date,`"%tdnn/dd/YY"')+`"-"'+string(date2,`"%tdnn/dd/YY"');
qui drop date date2;
qui encode QE, gen(QE_n);
label define QE_n 4 "FG", modify;
	
/*Reshape wide*/
qui reshape wide `LI_measures' DytmBP, i(sector QE* daterange) j(contractionary);
qui gen sector_n = cond(sector=="LI",1,cond(sector=="BHC",2,3));
qui sort sector_n QE_n;
qui drop sector_n;

/*************************************************************************************************************
7. Summary table by episode
*************************************************************************************************************/
/*Add blank rows and sector identifier rows*/
local LI_name `"Life insurers"';
local BHC_name `"Bank holding companies"';
local Market_name `"Market"';
qui set obs `=`=_N'+1';
qui gen order = _n;
foreach sector in LI BHC Market {;
	qui set obs `=`=_N'+2';
	qui sum order if sector==`"`sector'"';
	local break = r(min) - 1;
	qui replace order = `break' + 0.1 in `=`=_N'-1';
	qui replace order = `break' + 0.2 in `=_N';
	qui replace QE_n = r(min)*10 in `=_N';
	label define QE_n `=r(min)*10' `"\multicolumn{2}{l}{``sector'_name'}"', modify;
};
qui sort order;

foreach var of varlist Stock* Bond* {;
	qui replace `var' = `var' * 100;
};
qui format Stock* Bond* Dytm* CDS* %9.1f;

local nmeasures = `LI_nmeasures' + 1;
local 2nmeasures = `nmeasures'*2;
local filename event-study-summary;
local notes `" 
Notes:\\ 
a. Sum over announcement dates of the change in the on-the-run five year Treasury note during announcement window covering two minutes before to eighteen minutes after announcement, basis points.\\
b. Sum over announcement dates of the median change in the log stock price during announcement window covering two minutes before to eighteen minutes after announcement, log points. For panel Market, universe is firms in the S\&P 500 on the announcement date.\\
c. For panels Life Insurers and Bank Holding Companies, sum over announcement dates of the median change in the CDS spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), 5 year tenor, basis points.
For panel Market, sum over announcement dates of the log change in the on-the-run North American `CDX_sector' CDX index, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm).\\
d. Sum over announcement dates of the median change in the log bond price during announcement window covering at least two minutes before to eighteen minutes after announcement, log points.\\
"'; 
local title `"Event study effects of unconditional monetary policy"';
local headlines1 `""&&\multicolumn{`nmeasures'}{c}{Expansionary surprises}& \multicolumn{`nmeasures'}{c}{Contractionary surprises}\\ \cmidrule(l{.75em}){3-`=`nmeasures'+2'} \cmidrule(lr{.75em}){`=`nmeasures'+3'-`=`2nmeasures'+2'}" "';
local headlines2 `"" &Date range& `LI_headers' & `LI_headers'\\ ""';
local si_fmt `"3.2"';
listtex QE_n daterange `LI_vars0' `LI_vars1' using `"$PaperPath/`filename'.tex"', 
replace rstyle(tabular) headline("\begin{table}[htb]\centering" "\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\sisetup{table-format=`si_fmt'}" "\caption{`title'}" "\begin{tabularx}{\textwidth}{ll *{`2nmeasures'}{S}}" "\\" "\hline \hline" "\\" `headlines1' `headlines2' "\hline") footline("\hline \hline" "\multicolumn{`=`2nmeasures'+2'}{l}{" "\begin{minipage}{\hsize}" "\rule{0pt}{9pt}" "\footnotesize" "" "\noindent" "`notes'"  "\end{minipage}" "}\\" "\end{tabularx}" "\label{tab:`filename'}" "\end{table}"); 


