clear all
set more off
#delimit;
tempfile data;

/*************************************************************************************************************
This do-file estimates betas for life insurance companies and banks.
*************************************************************************************************************/
do paths;
local tenor 5;
local LI_name `"Insurance company"';
local BHC_name `"Bank"';
/*
/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
foreach sector in LI BHC {;

	/*Stock price data*/
	qui unzipfile "$DataPath/crspq_`sector'_dsf", replace;
	use crspq_`sector'_dsf, clear;
	erase crspq_`sector'_dsf.dta;
	qui tsset permno date; 
	qui gen adjprc = prc/cfacpr;
	tempfile dsf;
	save `dsf';
	qui unzipfile "$DataPath/CRSPQ_`sector'_stocknames", replace;
	qui use crspq_`sector'_stocknames;
	erase crspq_`sector'_stocknames.dta;
	qui egen maxdate = max(nameenddt), by(permno);
	qui keep if nameenddt==maxdate;
	qui drop maxdate;
	merge 1:m permno using `dsf', nogenerate keep(matched);
	qui drop if nameenddt<date;
	qui drop if inlist(comnam,"CANNON EXPRESS INC","INVERNESS MEDICAL TECHNOLOGY IN");
	qui egen maxout = max(shrout), by(permco date);
	qui drop if shrout!=maxout;
	qui drop maxout;
	tempfile `sector';
	qui save ``sector'', replace;
	
	/*CDS data*/
	qui unzipfile "$DataPath/markit_`sector'", replace;
	use markit_`sector', clear;
	erase markit_`sector'.dta;
	keep if tier==`"SNRFOR"';
	assert contributor=="Composite";
	qui egen numnotmissing = rownonmiss(spread*);
	drop if numnotmissing==0;
	duplicates report date ticker docclause ccy;
	/*After trimming observations with all spreads missing, date-ticker-docclause-ccy uniquely define observations*/
	drop if missing(spread`tenor');
	qui keep if ccy=="USD" & (docclause=="MR" | missing(docclause));
	
	merge 1:1 date ticker using ``sector'', nogenerate;
	capture drop sector;
	qui gen sector = `"`sector'"';
	qui save ``sector'', replace;
	
};

use `LI', replace;
append using `BHC';

/*Merge SP500*/
preserve;
use SP500, clear;
*freduse SP500, clear;
drop date;
rename daten date;
qui save `data', replace;
restore;
merge m:1 date using `data', nogenerate keep(matched) assert(matched using);

/*Merge CDX*/
preserve;
qui use  "$SavePath/CDX-Bloomberg" if rating==`"IG"' & source==`"CBGN"', clear;
qui egen maxseries = max(series), by(date);
qui keep if series==maxseries; /*Keeps on-the-run series*/
rename PX_LAST CDX;
qui keep date CDX;
qui save `data', replace;
restore;
merge m:1 date using `data', nogenerate keep(master matched);

qui drop if inlist(ticker,"MET","AMP","PFG") & sector=="BHC";
qui encode ticker, gen(id);
qui tsset id date;
qui gen hyear = hofd(date);
qui format hyear %th;

/*************************************************************************************************************
2. Estimate betas
*************************************************************************************************************/
foreach var of varlist adjprc SP500 CDX {;
	qui gen return_`var' = ln(`var'/L.`var');
	forvalues l = 2/4 {;
		qui replace return_`var' = ln(`var'/L`l'.`var') if missing(return_`var'); /*Holidays and weekends*/
	};
};
#delimit;
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		if inlist(`"${QE`q'_`i'}"',`"25nov2008 8:15:00"',`"12aug2009 14:16:10"',`"4nov2009 14:19:00"',`"27aug2010 10:00:00"',`"3nov2010 14:16:39"',`"31aug2012 10:00:00"',`"12dec2012 12:30:02"',`"18dec2013 14:00:00"') {; 
			local i = `i'+1;
			continue; /*Dates without any "action" in the Treasury market*/
		};
		else {;
			local datelist `"`datelist', `=td(`:word 1 of ${QE`q'_`i'}')' "';
			local i = `i'+1;
		};
	};
};

qui gen D_CDS = spread5y - L.spread5y;
qui gen marketcap = prc*shrout;
local Stock_spec `"reg return_adjprc return_SP500"';
local CDS_spec `"reg D_CDS return_CDX"';

qui drop if inlist(ticker,`"AIG"',`"VOYA"',`"PUK"'); /*AIG is weird for obvious reasons. VOYA only has data beginning in 2013. PUK is related to PRU.*/
qui levelsof ticker, local(tickers);
foreach measure in Stock CDS {;
	foreach var in beta se N {;
		qui gen `measure'_`var' = .;
	};
	foreach ticker of local tickers {;
		tokenize ``measure'_spec';
		forvalues t = `=th(2000h1)'/`=th(2013h2)' {;
			capture qui ``measure'_spec' if ticker==`"`ticker'"' & hyear==`t' & !inlist(date `datelist');
			if _rc==0 {;
				qui replace `measure'_beta = _b[`3'] if ticker==`"`ticker'"' & hyear==`t';
				qui replace `measure'_se = _se[`3'] if ticker==`"`ticker'"' & hyear==`t';
				qui replace `measure'_N = e(N) if ticker==`"`ticker'"' & hyear==`t';
			};
		};
	};
};

qui egen tag = tag(ticker hyear);
qui egen marketcapbar = mean(marketcap), by(ticker hyear);
qui compress ticker;
qui save betas, replace;

/*************************************************************************************************************
3. Figures
*************************************************************************************************************/
*/
use betas, clear;
qui egen totmarketcap = total(marketcapbar) if tag, by(sector hyear);
qui gen wgt = marketcapbar/totmarketcap;
local labsize medsmall;
local tfmt `"%tdCCYY-!Hh"';
forvalues yyyy = 2001(2)2013 {;
	local tlabels `"`tlabels' `=mdy(1,1,`yyyy')' "';
};

preserve;
qui keep if tag;
foreach measure in Stock CDS {;
	qui egen vw`measure' = total(wgt*`measure'_beta) if `measure'_N>=64, by(sector hyear);
	foreach sector in LI BHC {;
	
	
		if `"`sector'"'==`"LI"' | `"`measure'"'==`"CDS"' {;
			local colorj = 0; local symbolj = 0; local j = 0; local scatters; local legendkey; local legendorder;
			local colors `"blue red green black brown purple brown cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
			local symbols `"O D T S + X"';
			qui levelsof ticker if sector==`"`sector'"' & !missing(`measure'_beta), local(tickers);
			foreach ticker of local tickers {;
				local j = `j' + 1;
				foreach item in color symbol {;
					local `item'j = ``item'j' + 1;
					local `item': word ``item'j' of ``item's';
					if `"`item'"'==`""' {;
						local `item'j = 1;
						local `item': word ``item'j' of ``item's';
					};
				};
				local scatters `"`scatters' (scatter `measure'_beta date if ticker==`"`ticker'"' & `measure'_N>=64, msymbol(`symbol') msize(small) mcolor(`color'))"';
				local legendkey `"`legendkey' label(`j' `"`ticker'"')"';
				local legendorder `"`legendorder' `j'"';
			};
			local j = `j'+1;
			local legend `"on row(`=round(`j'/4)') rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) order(`legendorder') `legendkey'"';
		};
		else {;
			local scatters `"(scatter `measure'_beta date if `measure'_N>=64, msymbol(x) msize(small) mcolor(blue))"';
			local legend `"off"';
		};	
		
		twoway `scatters' (tsline vw`measure', lpattern(solid) lwidth(medthick) lcolor(black)) if tag & sector==`"`sector'"', subtitle("`measure' beta", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tlabels',tposition(outside) labsize(`labsize') format(`tfmt')) ysize($ysize_p) xsize($xsize_p);
		local filename = `"`sector'-`measure'-beta"';
		qui graph export `"$PresentationPath/`filename'.eps"', replace;
		local title = `"\hyperlink{`sector'-table}{``sector'_name' `=lower(`"`measure'"')' betas}"';
		local label `"`filename'"';
		TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
	};
};
restore;	
		
