clear all
set more off
#delimit;

/*************************************************************************************************************
This do-file does the MMF analysis at an annual level.

*************************************************************************************************************/
do Paths;
tempfile data;
local sector MMF;
local sector_name `"Money market fund"';

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
do MMF-analyzer-clean;

/*************************************************************************************************************
2. Collapse to data set of one observation per fund-year
*************************************************************************************************************/
qui encode MasterClassFund, gen(MasterClassFund_n); /*1 if not master class, 2 if master class*/
qui egen HasMaster = max(MasterClassFund_n), by(id_fund daten);
qui egen tag = tag(id_fund daten) if HasMaster==1;
qui keep if MasterClassFund=="True" | tag; /*Reduces data set to one observation per fund-date. I.e. keeps only one share class observation*/
qui drop tag; 
/*Annual means and standard deviation of variables*/
foreach var of varlist Folio* KS_holdingrisk WAMwk {;
	qui egen `var'_mu = mean(`var'), by(id_fund year);
	qui egen `var'_sd = sd(`var'), by(id_fund year);
};
qui egen tag = tag(id_fund year);
qui keep if tag; /*Reduces data set to one observation per fund-year*/
qui tsset id_fund year;

/*************************************************************************************************************
3. Scatterplots of fee adjustments
*************************************************************************************************************/
/*Scatterplot of charged and incurred expenses, 2006 and 2013*/
#delimit;
local sector MMF;
local sector_name `"Money market fund"';
local labsize medsmall;
local scatters_2006 `"(scatter FolioChgdExpenseRatiomo_mu FolioIncdExpenseRatiomo_mu if year==2006, msymbol(circle) msize(vsmall) mcolor(red))"';
local scatters_2013 `"(scatter FolioChgdExpenseRatiomo_mu FolioIncdExpenseRatiomo_mu if year==2013, msymbol(triangle) msize(vsmall) mcolor(blue))"';
local legend `"on row(2) rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) label(1 "2006") label(2 "2013")"';

/*Drop outlier for readability of graph*/
preserve;
foreach type in Incd Chgd {;
	sum Folio`type'ExpenseRatiomo_mu if regexm(FundName,`"Western Asset Inst US Treas Obl/Ins"'), meanonly;
	local `type'mean = substr(`"`=(round(r(mean),0.01))'"',1,3);
	local `type'mean = regexr(`"``type'mean'"',`"^\."',`"0."');
};
qui drop if regexm(FundName,`"Western Asset Inst US Treas Obl/Ins"');
local droplist `" To enhance readability, the figure omits one 2013 fund with 6 million under management, incurred expenses of `Incdmean' percent, and charged expenses of `Chgdmean' percent."';

twoway `scatters_2006' `scatters_2013', subtitle("Charged expenses as percent of assets", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("", size(`labsize')) xtitle("Incurred expenses as percent of assets", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize(3.5) xsize(6);
local filename `sector'-expenses-scatterplot; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.eps"', replace;
local title `"`sector_name' expense ratios, 2006 and 2013"';
local label `"fig:`filename'"';
local fullnote `"Notes:`droplist'"';
TexFigure `"`sector'_figures/`filename'.eps"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' figures.tex"', title(`sector' figures) author(Gabriel Chodorow-Reich) replace;
qui graph export `"$PaperPath/`filename'.eps"', replace;
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(tb) replace;

/*Presentation figure*/
local labsize medlarge;
local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',1);
twoway `scatters_2006' `scatters_2013', subtitle("Charged expenses as percent of assets", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("", size(`labsize')) xtitle("Incurred expenses as percent of assets", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
local title `"`sector_name' expense ratios"';
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(t) replace;

restore;


/*************************************************************************************************************
4. Table of summary statistics
a. Expense ratio falls. And distribution of expense ratios compresses.
b. Distribution of mean returns falls.
c. Time series variance of returns falls. This may be less interesting. For example, it is high in 2008 in part because the federal funds rate changes.
d. The net compound yield in 2013 is essentially zero. This means funds are setting their charged expense ratios to ensure a weakly positive return from
investing in the fund. This makes it plausible that they are actually eating operational expenses.
*************************************************************************************************************/
rename FolioChgdExpenseRatiomo_mu FolioChgdExpense_mu;
rename FolioIncdExpenseRatiomo_mu FolioIncdExpense_mu;
#delimit;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local netyield 7DCYwkly;
local stats `"mean sd p10 p50 p90 N"';

foreach var of varlist FolioChgdExpense_mu FolioIncdExpense_mu Folio_`grossyield'_mu Folio_`grossyield'_sd Folio_`netyield'_mu {;
	if `"`var'"'==`"Folio_`grossyield'_sd"' {;
		local `var' = regexr(`"`var'"',`"_sd$"',`"E_sd"'); /*Use SD of excess return*/
	};
	forvalues yyyy = 2006/2013 {;
		qui sum `var' if year==`yyyy', detail;
		foreach stat of local stats {;
			local `var'`yyyy'`stat' = r(`stat');
		};
	};
};

preserve; /*Data set of summary statistics*/
clear;
qui gen year = "";
foreach stat of local stats {;
	qui gen `stat' = .;
	local nstats = `nstats'+1;
	qui format `stat' %9.2f;
};
qui format N %9.0f;
local nvars = `nstats'+1;
foreach var in FolioChgdExpense_mu FolioIncdExpense_mu Folio_`grossyield'_mu Folio_`grossyield'_sd Folio_`netyield'_mu {;
	qui set obs `=`=_N'+2';
	qui replace year = `"`var'"' in `=_N';
	forvalues yyyy = 2006/2013 {;
		qui set obs `=`=_N'+1';
		qui replace year = `"`yyyy'"' in `=_N';
		foreach stat of local stats {;
			qui replace `stat' = ``var'`yyyy'`stat'' in `=_N';
		};
	};
};
qui replace year = "\multicolumn{`nvars'}{c}{Charged expense ratio, annual average}" if year=="FolioChgdExpense_mu";
qui replace year = "\multicolumn{`nvars'}{c}{Incurred expense ratio, annual average}" if year=="FolioIncdExpense_mu";
qui replace year = "\multicolumn{`nvars'}{c}{7 day gross simple yield, annual average}" if year=="Folio_`grossyield'_mu";
qui replace year = "\multicolumn{`nvars'}{c}{7 day gross simple yield, standard deviation}" if year=="Folio_`grossyield'_sd";
qui replace year = "\multicolumn{`nvars'}{c}{7 day net compound yield, annual average}" if year=="Folio_`netyield'_mu";

local filename `sector'-summary-statistics;
local notes `"Notes: The table reports cross-sectional statistics, by year, of time-series properties at the fund level."'; 
local title `"`sector_name' summary statistics"';
local headlines1 `""&\multicolumn{`nstats'}{c}{Statistic}\\ \cmidrule(lr{.75em}){2-`nvars'}" "';
local headlines2 `"" & \multicolumn{1}{c}{Mean} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{P10} & \multicolumn{1}{c}{P50} & \multicolumn{1}{c}{P90} & \multicolumn{1}{c}{Obs.} \\ ""';
local si_fmt `"5.5"';
listtex year `stats' using `"$OutputPath/`sector'_figures/`filename'.tex"', 
replace rstyle(tabular) headline("\begin{table}[htb]\centering" "\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\sisetup{table-format=`si_fmt'}" "\caption{`title'}" "\begin{tabularx}{\textwidth}{c *{`nstats'}{S}}" "\\" "\hline \hline" "\\" `headlines1' `headlines2' "\hline") footline("\hline \hline" "\multicolumn{`nvars'}{l}{" "\begin{minipage}{\hsize}" "\rule{0pt}{9pt}" "\footnotesize" "" "\noindent" "`notes'"  "\end{minipage}" "}\\" "\end{tabularx}" "\label{tab:`filename'}" "\end{table}"); 
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' figures.tex"', title(`sector' figures) author(Gabriel Chodorow-Reich) append;
/*Paper table*/
qui drop if inlist(year,"2007","2008","2009","2010","2011","2012");
listtex year `stats' using `"$PaperPath/`filename'.tex"', 
replace rstyle(tabular) headline("\begin{table}[tb]\centering" "\def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\sisetup{table-format=`si_fmt'}" "\caption{`title'}" "\begin{tabularx}{\textwidth}{c *{`=`nstats'-1'}{S} c}" "\\" "\hline \hline" `headlines1' `headlines2' "\hline") footline("\hline \hline" "\multicolumn{`nvars'}{l}{" "\begin{minipage}{\hsize}" "\rule{0pt}{9pt}" "\footnotesize" "\noindent" "`notes'"  "\end{minipage}" "}\\" "\end{tabularx}" "\label{tab:`filename'}" "\end{table}"); 

restore;

/*************************************************************************************************************
5. Exit regressions
*************************************************************************************************************/
#delimit;
estimates drop _all;
local sector MMF;
local sector_name `"Money market fund"';
local control_list `"TaxFree Prime AnyInstitutional"';
capture qui drop begin end exit Expenses2005 Instrument*;

preserve;

qui egen Expenses2005 = max(FolioIncdExpense_mu * (year==2005)), by(id_fund);
forvalues yyyy = 2006/2013 {;
	qui gen Instrument`yyyy'_2005 = Expenses2005 * (year==`yyyy');
};
qui gen begin = year;
qui gen end = year;
qui gen exit = !missing(FolioIncdExpense_mu) & missing(F.FolioIncdExpense_mu) & year!=2013;
stset year, id(id_fund) failure(exit);

restore;

/*************************************************************************************************************
6. Regressions
*************************************************************************************************************/
#delimit;
estimates drop _all;
capture drop *Expenses20??;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
/*Include sponsor FE?*/
*local sponsor_FE i.id_complex;

foreach type in Chgd Incd {;
	forvalues yyyy = 2006/2013 {;
		qui gen `type'Expenses`yyyy'_1 = Folio`type'Expense_mu if year==`yyyy';
		qui egen `type'Expenses`yyyy' = mean(`type'Expenses`yyyy'_1), by(id_fund);
		qui drop `type'Expenses`yyyy'_1;
	};
};

foreach measure in Folio_`grossyield'_mu Folio_`grossyield'E_sd KS_holdingrisk_mu WAMwk_mu {;	

	local controls `"TaxFree Prime AnyInstitutional"';
	local varlabels `"TaxFree `"Tax free"' AnyInstitutional `"Has institutional classes"' FolioIncdExpense_mu `"Incurred expenses (IV with 2006 value)"'"';

	local condition;
	if `"`measure'"'==`"KS_holdingrisk_mu"' {;
		local condition `" & Prime"';
		local controls = regexr(`"`controls'"',`"Prime"',`""');
	};
	
	qui eststo: ivregress 2sls `measure' `controls' `sponsor_FE' (FolioIncdExpense_mu = IncdExpenses2006) if year==2013 `condition', cluster(id_complex);
	qui estat first;
	matrix first = r(singleresults);
	qui estadd scalar Fstat = first[1,4];
	
	local nregs = `nregs' + 1;
};
if `"`sponsor_FE'"'==`"i.id_complex"' {;
	local indicate `"indicate(`"Fund sponsor FE = *.id_complex"', labels("\multicolumn{1}{c}{Yes}" "\multicolumn{1}{c}{No}"))"';		
};
else {;
	local indicate `""';
};
local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable: 2013}"'; 
local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
local groups2 `"& \multicolumn{1}{c}{mean} & \multicolumn{1}{c}{std. dev.} & \multicolumn{1}{c}{risky asset} & \multicolumn{1}{c}{average}"'; 
local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity (days)}"'; 
local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
local stats "Fstat N_clust N";
local stats_fmt "%9.1f %12.0fc %12.0fc";
local stats_label `" `"First stage F statistic"' `"Fund sponsor clusters"' `"Observations"' "';
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	
local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 3.5;
local title "`sector_name' reaching for yield regressions";
local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
local notes `"Notes: The mean and standard deviation yield refer to the mean and standard deviation of the weekly gross simple yield, expressed on an annualized basis. The risky asset allocation equals the annual average asset share in bank obligations less the shares in government securities and repo. The average maturity refers to the annual average of the maturity of a fund's securities. Regressions also include categorical variables for fund category (tax free, prime, or U.S. Treasury and agency securities and backed repo) and whether the fund has any institutional shares. Standard errors in parentheses and clustered by fund sponsor."';
local filename `"`sector'-reaching-annual-regressions`OUTPUT_PS'"';
esttab * using "$PaperPath/`filename'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) `indicate' drop(`controls' _cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot("\label{tab:`filename'}");	
estimates drop _all;
