clear all
set more off
#delimit;

/*************************************************************************************************************
This do-file does the MMF analysis.

*************************************************************************************************************/
do Paths;
tempfile data;
local sector MMF;
local sector_name `"Money market fund"';

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
local zipdata: dir `"$DataPath"' files `"*"';
if !regexm(`"`zipdata'"',`"mmf-analyzer-data.zip"') {;
	do MMF-analyzer-data;
};
qui unzipfile `"$DataPath/MMF-analyzer-data.zip"', replace;
qui use MMF-analyzer-data;
qui erase MMF-analyzer-data.dta;
qui drop if missing(ShareClassAssetswkmils);

qui encode FundComplex, gen(groupid);

/*************************************************************************************************************
2. Benchmark to Financial Accounts
Notes: Total assets match almost exactly. Treasury securities and repo are also almost identical. There are level differences
in Agencies and open market paper + time deposits. These have to do with classification. 
*************************************************************************************************************/
/*Convert portfolio holdings to billions of dollars and aggregate*/
qui gen ShareClassAssetswkbils = 10^(-3)*ShareClassAssetswkmils;
qui egen OpenMarketPaperAndTD = rowtotal(DomesticBankObligations TimeDeposits *CommercialPaper ABCP);
foreach var of varlist USTreasury-ABCP DemandNotesRated-AMTPaper OpenMarketPaperAndTD {;
	local l`var': variable label `var';
	local l`var' = regexr(`"`l`var''"',`" \(%\)$"',`""');
	qui gen `var'bils = 0.01*`var' * ShareClassAssetswkbils;
};
qui collapse (sum) *bils, by(daten);
foreach var of varlist * {;
	label variable `var' `"`l`var''"';
};
qui gen quarterly = qofd(daten);
qui egen lastdate = max(daten), by(quarterly);
qui keep if daten==lastdate;

/*Load data from FFA*/
preserve;
qui use quarterly FL63* using "$SavePath/FoF", clear; /*63 is sector code for MMFs*/
tempfile FA;
qui save `FA';
restore;
merge 1:1 quarterly using `FA', nogenerate keep(matched);
qui tsset quarterly, quarterly;

qui egen FLOpenMarketPaperAndTD = rowtotal(FL633030000 FL633069175);
label variable FLOpenMarketPaperAndTD `"ltab00d: Open Market Paper and Time Deposits"';
local sector MMF;
local labsize medsmall;
local tfmt `"%tqYY:!Qq"';
foreach pair in `"ShareClassAssetswkbils FL634090005Q"' `"USTreasury FL633061105"' `"USOther FL633061700"' `"OpenMarketPaperAndTD FLOpenMarketPaperAndTD"' `"Repos FL632050000"' {;
	tokenize `pair';
	local legend `"on row(2) rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) label(1 `"iMoneyNet"') label(2 `"FFA"') "';
	twoway (tsline `1' `2', lpattern(solid solid) lwidth(medthick medthick) lcolor(blue red)), subtitle("Billions of dollars", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
	local FFA_label: variable label `2';
	if regexm(`"`FFA_label'"',`"(^ltab[0-9]+d: )([a-zA-Z ]+)"') {;
		local short_label = regexs(2); /*Extract instrument description from FFA label*/
	};
	local filename = subinstr(`"`short_label'"',`" "',`"-"',.);
	qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
	local title = `"`short_label'"';
	local label `"fig:`filename'"';
	TexFigure `"`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
	local filenames `"`filenames' `filename'.tex"';
	local j = `j'+1;
	if `j'>12 {;
		local filenames `"`filenames' \clearpage"';
		local j = 0;
	};
	if `"`pair'"'==`"ShareClassAssetswkbils FL634090005Q"' {;
		qui graph export `"$PaperPath/`filename'.eps"', replace;
		local title `"`sector_name' `=lower(`"`short_label'"')'"';
		TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!htpb) replace;
	};		
};
qui EmbedTex `"`filenames'"' using `"$OutputPath/`sector'_figures/FFA-imoneynet benchmark figures.tex"', title(FFA and imoneynet) author(Gabriel Chodorow-Reich) replace;
