/*************************************************************************************************************
This ado-file constructs QE announcement window changes of stock prices.

TAQ:
g127 is always 0 after June 28, 2006
corr==0 excludes cancelled or corrected trades 

Options:
sectype: Maturity of Treasury
lineplots: Make figures with intraday line plot by institution
indexplots: Make figures with intraday line plot of index
famafrench: Run regression of daily return adjusting for Fama French factors
*************************************************************************************************************/
capture program drop EventStudyStock
program define EventStudyStock
version 12.1
syntax name(name=sector id="Sector LI, BHC, or SP500"), [lower(integer 2)] [upper(integer 18)] [sectype(integer 5)] [lineplots] [indexplots] [famafrench] [alldates]


clear
set more off
#delimit;
set matsize 2000;
tempfile data;


local LI_name `"Insurance company"';
local BHC_name `"Bank"';
local SP500_name `"S\&P 500"';
local LI_cluster `"cluster(date)"';
local BHC_cluster `"cluster(date)"';
local SP500_cluster `"cluster(date)"';
local measure `"stock"';
local idvar symbol;

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
/*TAQ data*/
if inlist(`"`sector'"',`"BHC"',`"LI"') {;
	qui unzipfile `"$DataPath/taq_`sector'"', replace;
	use taq_`sector';
	erase taq_`sector'.dta;
};
else if inlist(`"`sector'"',`"SP500"') {;
	qui unzipfile `"$DataPath/taq_`sector'_1"', replace;
	use taq_`sector';
	qui unzipfile `"$DataPath/taq_`sector'_2"', replace;
	append using taq_`sector';
	erase taq_`sector'.dta;
};

keep if corr==0;
drop if missing(cond) | inlist(cond,"O","Z"); /*O and z are trades with an incorrect time stamp*/

qui gen double datetime = Clock(string(date,"%tdCCYYMonDD")+substr(time,1,5),`"YMDhm"');
qui format datetime %tCDDmonCCYY_HH:MM;

qui collapse (mean) price (sum) size, by(symbol date datetime);

/*If SP500, drop banks and life insurers*/
if `"`sector'"'==`"SP500"' {;
	preserve;
	foreach other in LI BHC {;
		qui unzipfile `"$DataPath/taq_`other'"', replace;
		use taq_`other', clear;
		qui levelsof symbol, local(`other');
		erase taq_`other'.dta;
	};
	restore;
	foreach symbol in `LI' `BHC' {;
		qui drop if symbol==`"`symbol'"';
	};
};

qui drop if inlist(symbol,`"AIG"',`"VOYA"',`"PUK"'); /*AIG is weird for obvious reasons. VOYA only has data beginning in 2013. PUK is related to PRU.*/

/*Define panel*/
qui encode symbol, gen(id);
qui tsset id datetime, delta(1 minute);

qui gen logprice = log(price);

/*Merge yield to maturity of on the run 5 year Treasury*/
preserve;
qui use `"$DataPath/Treasuries-event-study-on-the-run"' if sectype==`sectype' & !missing(datetime), replace;
eventtime datetime, lower(`lower') upper(`upper') band(0:04:59) gen(eventtime);
qui replace eventtime =  1 if tin(10jul2013 17:25,10jul2013 17:29:00) & `upper'>40;
qui save `data'; 
restore;
merge m:1 datetime using `data', nogenerate assert(master matched using match_update) keep(master matched match_update) keepusing(ytm sectype eventtime) update;
qui egen iddate = group(id date);
qui tsset iddate datetime, delta(1 minute);
preserve;
qui use `data', replace;
qui collapse (mean) ytm, by(date eventtime);
qui save `data', replace; 
restore;

/*Merge data from CRSP daily stock file*/
preserve;
dsf `sector'; /*See program at end of file*/
tempfile dsf;
qui save `dsf', replace;
restore;
if `"`sector'"'==`"SP500"' {; /*Drop tickers from data set after they've left the S&P 500*/
	preserve;
	qui unzipfile "$DataPath/COMPM_SP500CONSTITUENTS";
	qui use COMPM_SP500CONSTITUENTS, clear;
	erase COMPM_SP500CONSTITUENTS.dta;
	qui gen ticker = ticx;
	qui gen date = datadate;
	qui keep ticker date;
	qui duplicates drop ticker date, force;
	merge 1:m ticker date using `dsf', nogenerate keep(matched);
	qui drop if namedt > date;
	qui save `dsf', replace; 
	restore;
};
qui gen ticker = symbol;
merge m:1 date ticker using `dsf', nogenerate keep(master matched);

qui gen QE = "";
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		if `"`alldates'"'==`""' & inlist(`"${QE`q'_`i'}"',`"25nov2008 8:15:00"',`"12aug2009 14:16:10"',`"4nov2009 14:19:00"',`"27aug2010 10:00:00"',`"3nov2010 14:16:39"',`"31aug2012 10:00:00"',`"12dec2012 12:30:02"',`"18dec2013 14:00:00"') {; 
			drop if date==dofC(`=tC(${QE`q'_`i'})');
			local i = `i'+1;
			continue; /*Dates without any "action" in the Treasury market*/
		};
		else {;
			qui replace QE = `"QE`q'_`i'"' if date==dofC(`=tC(${QE`q'_`i'})');
			local datelist `"`datelist' QE`q'_`i'"';
			local i = `i'+1;
		};
	};
}; 

/*************************************************************************************************************
2. Regressions
*************************************************************************************************************/
preserve;
/*Expand event time for November 25, 2008 and July 10, 2013 because it was before or after hours trading. 
Visual inspection of 5 year note confirms there were no other shocks.*/
qui replace eventtime = -1 if tin(10jul2013 16:15,10jul2013 16:43:00) | tin(25nov2008 07:45, 25nov2008 08:13);
qui replace eventtime =  1 if tin(10jul2013 16:58,10jul2013 19:00:00);
/*Collapse into event time*/
drop if missing(eventtime); 
capture tab date symbol;
qui collapse (mean) price logprice Dlogprice_2jan2008_24nov2008 ytm (sum) size, by(QE date eventtime symbol id iddate D1logprice marketcap);
merge m:1 date eventtime using `data', nogenerate update replace assert(matched using match_update match_conflict) keep(matched match_update match_conflict) keepusing(ytm);
/*So each window has same number of Treasury observations*/
qui tsset iddate eventtime, delta(1);
foreach var of varlist logprice* ytm {;
	qui gen D`var' = S2.`var';
};
qui gen double DytmBP = 10^4 * Dytm; /*Converts units to basis points*/
qui gen double DytmBP10 = 10^3 * Dytm; /*Converts units to 10 basis points*/
label variable DytmBP10 `"10 basis point change in `sectype' year Treasury"';
qui gen double DytmBP10_pos = (Dytm>0) if !missing(Dytm);
label variable DytmBP10_pos `"Increase in `sectype' year Treasury"';
qui gen double DytmBP10_neg = (Dytm<0) if !missing(Dytm);
label variable DytmBP10_neg `"Decrease in `sectype' year Treasury"';
qui gen double Dlogprice100 = 100 * Dlogprice;

local n = 0;
foreach rhs in `"c.DytmBP10#ibn.id"' `"DytmBP10"' `"DytmBP10_pos DytmBP10_neg"' `"DytmBP10 if substr(QE,3,1)!="1""' `"DytmBP10_pos DytmBP10_neg if substr(QE,3,1)!="1""' {;
	if `"`rhs'"'==`"c.DytmBP10#ibn.id"' & inlist(`"`sector'"',"BHC","SP500") {;
		continue; /*Too many BHCs to show estimates separately by BHC*/
	};
	
	local cons;
	if regexm(`"`rhs'"',`"^DytmBP10_pos DytmBP10_neg"') {;
		local cons `"noconstant"';
	};
	
	qui eststo: reg Dlogprice100 `rhs', `cons' robust ``sector'_cluster';
	local n = `n' + 1;	
		
	if `"`rhs'"'==`"c.DytmBP10#ibn.id"' {;
		local bnames: colnames e(b);
		foreach bname of local bnames {; /*Change factor variables to have labels in output*/
			if regexm(`"`bname'"',`"^[1-9][0-9]?[b]?[n]?\.id"') {; 
				if regexm(`"`bname'"',`"^[1-9][0-9]?bn\.id"') {; /*Stata is confused on whether bn belongs in the name. It is in colnames e(b), but not in the stored output.*/
					local bname = regexr(`"`bname'"',`"bn.id"',`".id"');
				};
				local id = regexr(`"`bname'"',`"\.id.*"',`""');
				local idlabel: label id `id';
				local cvarlabel `"10 b.p. change"';
			};
			local varlabels `"`varlabels' `bname' `"`idlabel' X `cvarlabel'"'"';
		};	
	};
	
	if strmatch(`"`rhs'"',`"* if *"') {;
		local indicate `"`indicate' & \multicolumn{1}{c}{No}"';
	};
	else {;
		local indicate `"`indicate' & \multicolumn{1}{c}{Yes}"';
	};		
};

local indicate `"`"Includes QE1 `indicate' \\"'"';
local groups `"& \multicolumn{`n'}{c}{Dependent variable: log change in stock price, p.p.}"'; 
local midrules `"\cmidrule(l{.75em}){2-`=`n'+1'}"';
local groups `"" `groups'\\ `midrules' ""';	
if e(N_clust)!=. {;
	local stats "r2 N_clust N";
	local stats_fmt "%9.3f %9.0f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Clusters"' `"Observations"' "';
};
else {;
	local stats "r2 N";
	local stats_fmt "%9.3f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Observations"' "';
};
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 6.6;
local title "``sector'_name' stock price response";
local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
local notes `"Notes: The dependent variable is the log change in the stock price from `lower' minutes before to `upper' minutes after the announcement. The variable change in `sectype' year Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury during the same window. Standard errors in parentheses."';
local filename `"`sector'-`measure'-response-window-`lower'-`upper'`OUTPUT_PS'"';
local filenames `"`filenames' `filename'"';
esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(a2)) se(par fmt(a2) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
EmbedTex `"`filename'"' using `"$OutputPath/`filename'-embedded.tex"', title(Stock price response) author(Gabriel Chodorow-Reich) replace;

/*Scatterplot with QE rounds labeled*/
local depvar Dlogprice;
local depvar_desc `"Log change in `measure' price"';
local labsize medsmall;
local colorj = 0; local symbolj = 0; local j = 0;
local colors `"blue red green brown"';
local symbols `"O D T S"';
forvalues q = 1/4 {;
	local j = `j' + 1;
	foreach item in color symbol {;
		local `item'j = ``item'j' + 1;
		local `item': word ``item'j' of ``item's';
		if `"`item'"'==`""' {;
			local `item'j = 1;
			local `item': word ``item'j' of ``item's';
		};
	};
	local scatters `"`scatters' (scatter `depvar' DytmBP if substr(QE,3,1)==`"`q'"', msymbol(`symbol') msize(vsmall) mcolor(`color'))"';
	local legendkey `"`legendkey' label(`j' `"QE `j'"')"';
};
local legend `"on row(`=round(`j'/2)') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
twoway `scatters', subtitle("`depvar_desc'", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures `lower' to `upper' minute window.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) replace;
estimates drop _all;

/*Mean of market cap*/
qui egen marketcapbar = mean(marketcap), by(`idvar');

/*Add combined dates for median, mean, value-weighted mean*/
qui levelsof `idvar', local(`idvar's);
foreach pair in `"16dec2008 18mar2009"' `"22may2013 19jun2013"' `"10jul2013 18sep2013"' {;
	local d = `d' + 1;
	tokenize `pair';
	foreach `idvar' of local `idvar's {;
		qui set obs `=`=_N'+1';
		qui replace date = `d' in `=_N';
		foreach var of varlist `depvar' DytmBP {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(sum) if r(N)!=0 in `=_N';
		};
		qui replace DytmBP = . in `=_N' if r(N)!=2;
		foreach var of varlist marketcap* {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(mean) in `=_N';
		};
		qui replace `idvar' = `"``idvar''"' in `=_N';
	};
};

/*Add median, mean, value-weighted mean, and p-values of 2-sided t-test on each date to data set*/
qui levelsof date, local(dates);
qui gen Pvalue = .;
foreach date of local dates {;
	foreach stat in Median Mean vwMean {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace symbol = `"`stat'"' in `=_N';
		sum DytmBP if date == `date', meanonly;
		qui replace DytmBP = r(mean) in `=_N';
		if `"`stat'"'==`"Median"' {; 
			qui sum `depvar' if date == `date' & !inlist(symbol,`"Median"',`"Mean"',`"vwMean"'), detail;
			qui replace `depvar' = r(p50) in `=_N';
		};
		else if `"`stat'"'==`"Mean"' {;
			local wgt `""';
		};
		else if `"`stat'"'==`"vwMean"' {;
			local wgt `"[aw=marketcapbar]"';
		};
		if `"`sector'"'!=`"SP500"' | `date'<=td(31dec2013) {; 
			foreach vce in "" ", robust" {;
				capture qui reg `depvar' if date==`date' & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"') `wgt' `vce';
				if _rc==0 {;
					qui replace `depvar' = _b[_cons] in `=_N';
					qui test _cons;
					/*Use maximum of conventional and robust SE*/
					qui replace Pvalue = r(p) in `=_N' if (r(p)>Pvalue[`=_N'] & !missing(r(p))) | missing(Pvalue[`=_N']);
				};
			};
		};
	};
};	

/*Drop combined dates with individual tickers*/
qui drop if inrange(date,1,3) & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"');

/*Drop combined dates with summary measures and add back later*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		foreach var of varlist `depvar' DytmBP Pvalue {;
			sum `var' if `idvar'==`"`stat'"' & date==`date', meanonly;
			local `stat'`var'`date' = r(mean);
		};
	};
};
qui drop if inrange(date,1,3);

/*Scatterplot with dates labeled*/
local depvar Dlogprice;
local depvar_desc `"Log change in `measure' price"';
local labsize medsmall;
local altangle1 75; /*Put some date labels at an alternate label to enhance readability*/
local altangle2 105; /*Put some date labels at an alternate label to enhance readability*/
qui levelsof DytmBP, local(values);
foreach value of local values {;
	sum date if round(DytmBP,0.0001)==round(`value',0.0001), meanonly;
	if inlist(r(mean),`=td(9aug2011)',`=td(1dec2008)',`=td(13sep2012)') {;
		local tlabels`altangle1' `"`tlabels`altangle1'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else if inlist(r(mean),`=td(10aug2010)',`=td(18sep2013)',`=td(23sep2009)',`=td(22may2013)') {;
		local tlabels`altangle2' `"`tlabels`altangle2'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else {;
		local tlabels `"`tlabels' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
};
local xlabsize small; 
local xlabels `"xlabel(`tlabels', tposition(outside) labsize(`xlabsize') angle(vertical))"';
foreach a in `altangle1' `altangle2' {;
	local xlabels `"`xlabels' xlabel(`tlabels`a'', tposition(outside) labsize(`xlabsize') angle(`a') custom add)"';
};
if `"`sector'"'==`"LI"' {;
	local colorj = 0; local symbolj = 0; local j = 0; local scatters; local legendkey;
	local colors `"blue red green black brown purple brown cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
	local symbols `"O D T S + X"';
	qui levelsof symbol if !inlist(symbol,`"Median"',`"Mean"',`"vwMean"'), local(tickers);
	foreach ticker of local tickers {;
		local j = `j' + 1;
		foreach item in color symbol {;
			local `item'j = ``item'j' + 1;
			local `item': word ``item'j' of ``item's';
			if `"`item'"'==`""' {;
				local `item'j = 1;
				local `item': word ``item'j' of ``item's';
			};
		};
		local scatters `"`scatters' (scatter `depvar' DytmBP if symbol==`"`ticker'"', msymbol(`symbol') msize(small) mcolor(`color') xaxis(1 2))"';
		local legendkey `"`legendkey' label(`j' `"`ticker'"')"';
	};
	local j = `j'+1;
	local legend `"on row(`=round(`j'/3)') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
};
else {;
	local scatters `"(scatter `depvar' DytmBP, msymbol(x) msize(small) mcolor(blue) xaxis(1 2))"';
	local scatters `"`scatters' (scatter `depvar' DytmBP if symbol==`"Median"', msymbol(O) msize(medium) mcolor(black) xaxis(1 2))"';
	local legend `"on col(1) rowgap(0) size(`labsize') position(7) region(lcolor(white)) ring(0) order(2) label(2 "Median")"';
};
twoway `scatters', yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot2-window-`lower'-`upper'`alldates'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
local fullnote `"Notes: The log change in stock price is from `lower' minutes before to `upper' minutes after the announcement. The change in the on-the-run `sectype' year Treasury is the change in the yield to maturity during the same window."';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(fig:`label') note(`fullnote') position(pt) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures `lower' to `upper' minute window.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) append;

/*Presentation figure*/
local labsize medlarge;
local xlabsize medium; 
local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',1);
local xlabels = subinstr(`"`xlabels'"',`"small"',`"`xlabsize'"',.);
twoway `scatters', yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
if `upper'==18 {;
	local title `"\hyperlink{`sector' `measure' `lower' to 95 minute window}{``sector'_name' `measure' price response}"';
};
else {;
	local title `"\hyperlink{`sector' `measure' `lower' to 18 minute window}{``sector'_name' `measure' price response}"';
};
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;

/*Add back combined dates with summary measures*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace `idvar' = `"`stat'"' in `=_N';
		foreach var of varlist `depvar' DytmBP Pvalue {;
			qui replace `var' = ``stat'`var'`date'' in `=_N';
		};
	};
};


qui save "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'`alldates'", replace;


restore;

/*************************************************************************************************************
3. Figures of stock price by date
*************************************************************************************************************/
if `"`lineplots'"'==`"lineplots"' | `"`indexplots'"'==`"indexplots"' {;
 
qui gen knot = price if eventtime==0;
qui egen meanknot = mean(knot), by(id date);
qui gen dailyindex = 100*price/meanknot;
qui drop knot meanknot;
qui replace ytm = 100*ytm;

/*Generate value-weighted intraday stock price index*/
preserve; 
qui tsset iddate datetime, delta(1 minute);
qui tsfill;
qui replace date = dofC(datetime) if missing(date);
qui replace dailyindex = L.dailyindex if missing(dailyindex);
qui egen marketcapbar = mean(marketcap), by(`idvar');
qui collapse (mean) dailyindex ytm eventtime [aw=marketcapbar], by(date datetime);
rename dailyindex vw_dailyindex;
tempfile data;
save `data';
restore;

if `"`indexplots'"'==`"indexplots"' {;
	qui use `data', replace;
	rename vw_dailyindex dailyindex;
	local plottype indexplots;
	
	/*Merge S&P 500*/
	preserve;
	qui unzipfile `"$SavePath/SP500-by-minute-tickdata"', replace;
	qui use SP500-by-minute-tickdata, clear;
	erase SP500-by-minute-tickdata.dta;
	rename close SP500;
	keep datetime SP500;
	qui replace datetime = Cofc(datetime);
	tempfile tickdata;
	qui save `tickdata';
	restore; 
	merge 1:1 datetime using `tickdata', keep(master matched);
	qui egen knot = max(SP500 * (eventtime==0)), by(date);
	qui gen SP500index = 100*SP500/knot;
	qui drop knot;
};
else {;
	merge m:1 datetime using `data', assert(matched using) keep(matched) keepusing(vw_dailyindex) nogenerate;
	qui levelsof symbol, local(tickers);
	local plottype lineplots;
};
qui gen double time = datetime - Cofd(dofC(datetime));
qui format time %tCHH:MM:SS;
	
local labsize medsmall;
local tfmt `"%tCHH:MM"';
local filenames;
qui levelsof date, local(dates);
foreach date of local dates {;
	di `"`=string(`date',`"%td"')'"';
	if inlist(`date',`=td(25nov2008)',`=td(10jul2013)') {;
		continue;
	};
	sum datetime if date==`date' & eventtime==0, meanonly;
	local eventtime = r(mean);
	local lower = `eventtime' - msofminutes(2);
	local upper = `eventtime' + msofminutes(18);
	if `"`sector'"'==`"SP500"' {;
		local tlabels `"`=`eventtime'-msofminutes(20)' (`=msofminutes(10)') `=`eventtime'+msofminutes(40)'"';
		local ivalues `"97 (1) 103"';
	};
	else {;
		local tlabels `"`=tC(`=string(`date',`"%td"')' 10:00)' (`=msofminutes(60)') `=tC(`=string(`date',`"%td"')' 16:00)'"';
	};
	sum ytm if date==`date' & sectype==5, meanonly;
	if r(max)-r(min)<0.4 {;
		local topyval = 0.1*ceil(10*r(mean))+0.2; 
	};
	else {;
		local topyval = 0.1*ceil(10*r(max));
	};
	local yvalues `"`=`topyval'-0.5' (0.1) `topyval'"';
	if `"`lineplots'"'==`"lineplots"' {;
		foreach ticker of local tickers {;
			local legend `"on row(3) rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) label(1 `"`ticker'"') label(2 `"Value-weighted sector"') label(3 `"`sectype'y TNote (right axis)"')"';
			twoway (tsline dailyindex, lpattern(solid) lwidth(medthick) lcolor(blue)) (tsline vw_dailyindex, lpattern(longdash_dot) lwidth(medthick) lcolor(green)) (tsline ytm, lpattern(dash) lwidth(medthick) lcolor(red) yaxis(2)) if date==`date' & symbol==`"`ticker'"' & inrange(time,tC(09:15:00),tC(16:30:00)), tline(`lower', lpattern(dash)) tline(`upper', lpattern(dash)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize') labcolor(blue)) ytitle("Stock price, announcement time = 100", size(`labsize') color(blue)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize') labcolor(red) axis(2)) ytitle("Yield to maturity, `sectype' year Treasury", size(`labsize') color(red) axis(2)) ttitle("", size(`labsize')) tlabel(`tlabels',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
			local filename = `"`sector'-`ticker'-"' + string(`date',`"%td"');
			qui graph export `"$OutputPath/`sector'_figures/`filename'.eps"', replace;
			local title = `"Stock response of `ticker', "' + string(`date',`"%tdMonth_dd,_CCYY"');
			local label `"`filename'"';
			TexFigure `"`filename'.eps"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
			local filenames `"`filenames' `filename'.tex"';
		};
		local filenames `"`filenames' \clearpage"';
	};
	else {;
		local legend `"on col(3) rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) label(1 `"`sector'"') label(2 `"S&P 500"') label(3 `"`sectype'y TNote (right axis)"')"';
		twoway (tsline dailyindex SP500index, lpattern(solid longdash_dot) lwidth(medthick medthick) lcolor(blue green)) (tsline ytm, lpattern(dash) lwidth(medthick) lcolor(red) yaxis(2)) if date==`date' & inrange(time,tC(09:30:00),tC(16:00:00)), tline(`lower', lpattern(dash)) tline(`upper', lpattern(dash)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`ivalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize') labcolor(blue)) ytitle("Stock price, announcement time = 100", size(`labsize') color(blue)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize') labcolor(red) axis(2)) ytitle("Yield to maturity, `sectype' year Treasury", size(`labsize') color(red) axis(2)) ttitle("", size(`labsize')) tlabel(`tlabels',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
		local filename = `"`sector'-"' + string(`date',`"%td"');
		qui graph export `"$OutputPath/`sector'_figures/`filename'.eps"', replace;
		local title = `"Stock response, "' + string(`date',`"%tdMonth_dd,_CCYY"');
		local label `"`filename'"';
		TexFigure `"`filename'.eps"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
		local filenames `"`filenames' `filename'.tex"';
	};	
};

qui EmbedTex `"`filenames'"' using `"$OutputPath/`sector'_figures/Stock price response `plottype'.tex"', title(Stock price response) author(Gabriel Chodorow-Reich) replace;

};


/*************************************************************************************************************
4. Fama-French adjusted results
*************************************************************************************************************/
if `"`famafrench'"'==`"famafrench"' {;

/*Expand event time for November 25, 2008 and July 10, 2013 because it was before or after hours trading. 
Visual inspection of 5 year note confirms there were no other shocks.*/
qui replace eventtime = -1 if tin(10jul2013 16:15,10jul2013 16:43:00) | tin(25nov2008 07:45, 25nov2008 08:13);
qui replace eventtime =  1 if tin(10jul2013 16:58,10jul2013 19:00:00);
/*Collapse into event time*/
drop if missing(eventtime); 
capture tab date symbol;
qui collapse (mean) price logprice ytm (sum) size, by(QE date eventtime symbol id iddate D1logprice permno);
merge m:1 date eventtime using `data', nogenerate update replace assert(matched using match_update match_conflict) keep(matched match_update match_conflict) keepusing(ytm);
/*So each window has same number of Treasury observations*/
qui tsset iddate eventtime, delta(1);
foreach var of varlist logprice ytm {;
	qui gen D`var' = S2.`var';
};
qui keep if eventtime==1;
qui drop if missing(permno);

/*Merge data from CRSP daily stock file*/
preserve;
dsf `sector'; /*See program at end of file*/
tempfile dsf;
qui save `dsf', replace;
restore;
merge 1:1 date permno using `dsf', assert(matched using) nogenerate;
qui egen idnew = mean(id), by(permno);
assert id==idnew | missing(id);
qui replace id = idnew;
qui drop if missing(id);

/*Merge Fama-French factors*/
preserve;
FamaFrench daily;
tempfile FF;
save `FF';
restore;
merge m:1 date using `FF', nogenerate;

/*Define categorical events*/
qui gen PosSurprise = Dytm<0;
qui gen NegSurprise = Dytm>0 & !missing(Dytm); 

/*Define dates*/
qui levelsof date if !missing(eventtime), local(dates);
foreach date of local dates {;
	qui gen _`date' = date==`date';
	label variable _`date' `"`=string(`date',`"%tdMonth_dd,_CCYY"')'"';
	local eventdates `"`eventdates' _`date'"';
};

/*Run regressions*/
estimates drop _all;
qui gen Dlogprice100 = 100*Dlogprice;
qui gen D1logprice100 = 100*D1logprice;
qui eststo: reg Dlogprice100  `eventdates' if !missing(eventtime), noconstant cluster(id);
qui estadd local FE `"No"';
qui estadd local FFload `"No"';
qui eststo: reg D1logprice100 `eventdates' if !missing(eventtime), noconstant cluster(id);
qui estadd local FFload `"No"';
qui estadd local FE `"No"';
qui eststo: reg D1logprice100 `eventdates' ibn.id ibn.id#c.Mkt_RF ibn.id#c.SMB ibn.id#c.HML if date>=td(1jan2008), noconstant cluster(id);
qui estadd local FFload `"Yes"';
qui estadd local FE `"Yes"';

/*Output regression table*/
local nregs = 3;
local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable: log change in stock price}"'; 
local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
local groups2 `"& \multicolumn{1}{L}{intraday} & \multicolumn{1}{L}{daily}  & \multicolumn{1}{L}{daily}"'; 
local midrules2 `"\cmidrule(lr{.75em}){2-2} \cmidrule(lr{.75em}){3-3} \cmidrule(lr{.75em}){4-4}"'; 
local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`midrules2'" "';	
local stats "FE FFload N_clust N";
local stats_fmt "%3s %3s %12.0fc %12.0fc";
local stats_label `" `"Institution FE"' `"Institution-specific Fama-French loadings"' `"Institution clusters"' `"Observations"' "';
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 1.3;
local title "``sector'_name' robustness regressions";
local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
local notes `"Notes: Standard errors in parentheses and clustered by institution. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
local filename `"`sector'-robustness-regressions`OUTPUT_PS'"';
esttab * using "$OutputPath/`filename'.tex",  replace cells(b(star fmt(%9.2f)) se(par fmt(%9.2f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) keep(`eventdates', relax) drop(`droplist', relax) `prehead' `posthead' order(`order') label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx {1}{c} {1}{L} "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot("\label{tab:`filename'}");	
estimates drop _all;
EmbedTex `"`filename'"' using `"$OutputPath/`sector'-robustness-regressions-embedded.tex"', title(``sector'_name' robustness regressions) author(Gabriel Chodorow-Reich) replace;
copy "$OutputPath/`filename'.tex" "$PaperPath/`filename'.tex", replace;


};


end;



/*************************************************************************************************************
5. Called programs
*************************************************************************************************************/
capture program drop dsf;
program define dsf;
syntax name(name=sector id="Sector LI, BHC, or SP500");
qui unzipfile "$DataPath/crspq_`sector'_dsf", replace;
use crspq_`sector'_dsf, clear;
erase crspq_`sector'_dsf.dta;
qui tsset permno date;
qui replace prc = -prc if prc<0; /*Negative price indicates observation is midpoint between bid and ask*/
qui gen double D1logprice = log(prc) - log(L.prc) if cfacpr==L.cfacpr; /*1 day change*/
qui replace D1logprice = log(prc) - log(L3.prc) if missing(D1logprice) & dow(date)==1 & cfacpr==L3.cfacpr; /*If event date on Monday, use prior Friday for prior observation*/
qui gen double marketcap = prc*shrout;
#delimit;
foreach pair in "2jan2008 24nov2008" {;
	tokenize `pair';
	forvalues s = 1/2 {;
		qui egen adjprc_``s'' = max(prc/cfacpr * (date==td(``s''))), by(permno);
	};
	qui gen double Dlogprice_`1'_`2' = log(adjprc_`2'/adjprc_`1');
};
tempfile dsf;
save `dsf';
qui unzipfile "$DataPath/CRSPQ_`sector'_stocknames", replace;
qui use crspq_`sector'_stocknames;
erase crspq_`sector'_stocknames.dta;
qui egen maxdate = max(nameenddt), by(permno);
qui keep if nameenddt==maxdate;
qui drop maxdate;
merge 1:m permno using `dsf', nogenerate keep(matched);
qui drop if nameenddt<date;
qui drop if inlist(comnam,"CANNON EXPRESS INC","INVERNESS MEDICAL TECHNOLOGY IN");
qui egen maxout = max(shrout), by(permco date);
qui drop if shrout!=maxout;
qui drop maxout;
end;
