/*************************************************************************************************************
This ado-file constructs QE announcement window changes of CDX prices.


Options:
sectype: Maturity of Treasury
lineplots: Make figures with intraday line plot by institution
*************************************************************************************************************/
capture program drop EventStudyCDX
program define EventStudyCDX
version 12.1
syntax name(name=sector id="Sector IG or HY"), [lower(integer 2)] [upper(integer 18)] [sectype(integer 5)] [tenor(integer 5)] [lineplots]


clear
set more off
#delimit;
tempfile data;


local IG_name `"Investment grade"';
local HY_name `"High yield"';
local measure `"CDX"';

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
use "$SavePath/CDX-Bloomberg" if rating==`"`sector'"', clear;
qui keep date series source PX_LAST;
rename PX_LAST CDX;
qui reshape wide CDX, i(date series) j(source) string;
qui tsset series date;
qui gen DlogintradayNL = log(CDXCBGN/CDXCBGL);
qui gen DlogintradayNT = log(CDXCBGN/CDXCBGT);
qui gen DlogintradayLT = log(CDXCBGL/CDXCBGT);

qui gen double Dlogprice = ln(CDXCBGN/L.CDXCBGN); /*1 day change*/
qui replace Dlogprice = ln(CDXCBGN/L3.CDXCBGN) if missing(Dlogprice) & dow(date)==1; /*If event date on Monday, use prior Friday for prior observation*/
foreach var of varlist Dlog* {;
	qui gen `var'100 = 100*`var';
};

qui gen QE = "";
qui gen time = .;
qui format time %tCHH:MM;
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		if inlist(`"${QE`q'_`i'}"',`"25nov2008 8:15:00"',`"12aug2009 14:16:10"',`"4nov2009 14:19:00"',`"27aug2010 10:00:00"',`"3nov2010 14:16:39"',`"31aug2012 10:00:00"',`"12dec2012 12:30:02"',`"18dec2013 14:00:00"') {; 
			local i = `i'+1;
			continue; /*Dates without any "action" in the Treasury market*/
		};
		else {;
			qui replace QE = `"QE`q'_`i'"' if date==dofC(`=tC(${QE`q'_`i'})');
			local time: word 2 of ${QE`q'_`i'};
			qui replace time = tC(`time') if QE == `"QE`q'_`i'"';
			local datelist `"`datelist' QE`q'_`i'"';
			local i = `i'+1;
		};
	};
};

qui gen Dlogintraday100 = DlogintradayNL100 if time>tC(12:15:00) & !missing(time);
qui replace Dlogintraday100 = DlogintradayLT100 if time<tC(12:15:00);
/*CBGN, CBGL, CBGT report data as of 17:15 in New York, London, and Tokyo, respectively. For the dates I use,
I don't need to worry about differences in daylight savings time adjustment dates*/

/*Denote eventdate for on-the-run index for each date. The on-the-run is the most liquid -- see p.9 of
http://www.markit.com/assets/en/docs/products/data/indices/credit-index-annexes/Credit_Indices_Primer_Apr_2013.pdf*/
qui gen eventdate = .;
foreach date of local datelist {;
	qui sum series if date==dofC(`=tC(${`date'})');
	qui replace eventdate = 1 if date==dofC(`=tC(${`date'})') & series==r(max);
};

/*Merge change in yield to maturity of on the run 5 year Treasury. Use intraday window for Treasury because noise in RHS measure would introduce attenuation bias*/
preserve;
qui use `"$DataPath/Treasuries-event-study-on-the-run"' if sectype==`sectype' & !missing(datetime), replace;
eventtime datetime, lower(`lower') upper(`upper') band(0:04:59) gen(eventtime);
qui replace eventtime =  1 if tin(10jul2013 17:25,10jul2013 17:29:00) & `upper'>40;
qui drop if missing(eventtime);
qui collapse (mean)  ytm, by(date eventtime);
qui tsset date eventtime, delta(1);
foreach var of varlist ytm {;
	qui gen D`var' = S2.`var';
};
qui keep if eventtime == 1;
qui gen DytmBP = 10^4 * Dytm; /*Converts units to basis points*/
qui gen DytmBP10 = 10^3 * Dytm; /*Converts units to 10 basis points*/
label variable DytmBP10 `"10 basis point change in `sectype' year Treasury"';
qui gen double DytmBP10_pos = (Dytm>0) if !missing(Dytm);
label variable DytmBP10_pos `"Increase in `sectype' year Treasury"';
qui gen double DytmBP10_neg = (Dytm<0) if !missing(Dytm);
label variable DytmBP10_neg `"Decrease in `sectype' year Treasury"';
qui save `data';
restore;
merge m:1 date using `data', nogenerate keep(master matched) keepusing(*ytm*);


/*************************************************************************************************************
2. Regressions
*************************************************************************************************************/
preserve;
qui keep if eventdate==1;
local n = 0;
foreach depvar in Dlogprice100 Dlogintraday100 {;
	foreach rhs in `"DytmBP10"' `"DytmBP10_pos DytmBP10_neg"' `"DytmBP10 if substr(QE,3,1)!="1""' `"DytmBP10_pos DytmBP10_neg if substr(QE,3,1)!="1""' {;
		
		local cons;
		if regexm(`"`rhs'"',`"^DytmBP10_pos DytmBP10_neg"') {;
			local cons `"noconstant"';
		};
		
		qui eststo: reg `depvar' `rhs', `cons' robust;
		
		local n = `n' + 1;	

		if strmatch(`"`rhs'"',`"* if *"') {;
			local indicate `"`indicate' & \multicolumn{1}{c}{No}"';
		};
		else {;
			local indicate `"`indicate' & \multicolumn{1}{c}{Yes}"';
		};		
	};
};

local indicate `"`"Includes QE1 `indicate' \\"'"';
local groups1 `"& \multicolumn{`n'}{c}{Dependent variable: log change in CDX `sector' index}"'; 
local midrules1 `"\cmidrule(l{.75em}){2-`=`n'+1'}"';
local groups2 `"& \multicolumn{`=`n'/2'}{c}{1 day change} & \multicolumn{`=`n'/2'}{c}{Intraday change}"'; 
local midrules2 `"\cmidrule(l{.75em}){2-`=`n'/2+1'} \cmidrule(l{.75em}){`=`n'/2+2'-`=`n'+1'}"';
local groups `"" `groups1'\\ `midrules1' `groups2'\\ `midrules2' ""';	
if e(N_clust)!=. {;
	local stats "r2 N_clust N";
	local stats_fmt "%9.3f %9.0f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Clusters"' `"Observations"' "';
};
else {;
	local stats "r2 N";
	local stats_fmt "%9.3f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Observations"' "';
};
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 3.3;
local title "``sector'_name' CDX response, p.p.";
local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
local notes `"Notes: The dependent variable is the log change in the on-the-run CDX North American `sector' index. The variable change in `sectype' year Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury from `lower' minutes before to `upper' minutes after the announcement. Heteroskedastic robust standard errors in parentheses."';
local filename `"`sector'-CDX-response-window-`lower'-`upper'`OUTPUT_PS'"';
local filenames `"`filenames' `filename'"';
esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(%9.1f)) se(par fmt(%9.1f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
EmbedTex `"`filename'"' using `"$OutputPath/`filename'-embedded.tex"', title(CDS price response) author(Gabriel Chodorow-Reich) landscape replace;
estimates drop _all;

/*Scatterplot with QE rounds labeled*/
local depvar Dlogintraday100;
local devar_desc `"Log change in CDX, p.p."';
local labsize medsmall;
local colorj = 0; local symbolj = 0; local j = 0;
local colors `"blue red green brown"';
local symbols `"O D T S"';
forvalues q = 1/4 {;
	local j = `j' + 1;
	foreach item in color symbol {;
		local `item'j = ``item'j' + 1;
		local `item': word ``item'j' of ``item's';
		if `"`item'"'==`""' {;
			local `item'j = 1;
			local `item': word ``item'j' of ``item's';
		};
	};
	local scatters `"`scatters' (scatter `depvar' DytmBP if substr(QE,3,1)==`"`q'"', msymbol(`symbol') msize(vsmall) mcolor(`color'))"';
	local legendkey `"`legendkey' label(`j' `"QE `j'"')"';
};
local legend `"on row(`=round(`j'/2)') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
twoway `scatters', subtitle("`depvar_desc'", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot-window-`lower'-`upper'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) replace;

/*Scatterplot with dates labeled*/
qui levelsof DytmBP, local(values);
foreach value of local values {;
	sum date if round(DytmBP,0.0001)==round(`value',0.0001), meanonly;
	local tlabels `"`tlabels' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
};
local scatters `"(scatter `depvar' DytmBP, msymbol(O) msize(small) mcolor(blue) xaxis(1 2))"';
local legend `"off"';
local scatters `"`scatters'"';
twoway `scatters', yline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) xlabel(`tlabels',tposition(outside) labsize(`labsize') angle(vertical)) xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot2-window-`lower'-`upper'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
local fullnote `"Notes: The change in CDX is the one day change in the on-the-run CDX `sector' index, on the announcement date. The change in `sectype' year Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury from `lower' minutes before to `upper' minutes after the announcement.`dropdates'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(fig:`label') note(`fullnote') position(!ht) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) append;

/*Add combined dates*/
local depvar Dlogintraday100;
foreach pair in `"16dec2008 18mar2009"' `"22may2013 19jun2013"' `"10jul2013 18sep2013"' {;
	local d = `d' + 1;
	tokenize `pair';
	qui set obs `=`=_N'+1';
	qui replace date = `d' in `=_N';
	qui sum `depvar' if inlist(date,td(`1'),td(`2'));
	qui replace `depvar' = r(sum) if r(N)!=0 in `=_N';
};

qui gen Median="Median";
qui gen Pvalue=.;

qui save "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'", replace;
restore;

/*************************************************************************************************************
4. Figures of CDS over medium horizon
*************************************************************************************************************/
if `"`lineplots'"'==`"lineplots"' {;

local labsize medsmall;
local tfmt `"%tdMon-YY"';
local legend off;
local filenames;
local tlabels;
qui levelsof CDX_code_short, local(tickers);
forvalues yyyy = 2008/2013 {;
	local tlabels `"`tlabels' `=td(1jan`yyyy')'"';
};
local j = 0;
foreach ticker of local tickers {;
	twoway (tsline CDXCBGN, lpattern(solid) lwidth(medthick) lcolor(blue)) if CDX_code_short==`"`ticker'"', subtitle("CDX index", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tlabels',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
	local filename = `"`sector'-`measure'-`ticker'"';
	qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
	local title = `"`ticker' `measure'"';
	local label `"`filename'"';
	TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
	local filenames `"`filenames' `sector'_figures/`filename'.tex"';
	local j = `j'+1;
	if `j'>12 {;
		local filenames `"`filenames' \clearpage"';
		local j = 0;
	};
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector' `measure' price response figures.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) append;

};

end;
