/*************************************************************************************************************
This ado-file constructs QE announcement window changes of bond prices.

TAQ:
g127 is always 0 after June 28, 2006
corr==0 excludes cancelled or corrected trades 

Options:
sectype: Maturity of Treasury
lineplots: Make figures with intraday line plot by institution
*************************************************************************************************************/
capture program drop EventStudyBond
program define EventStudyBond
version 12.1
syntax name(name=sector id="Sector LI or BHC"), [lower(integer 2)] [upper(integer 18)] [sectype(integer 5)] [lineplots] [yield] [alldates]


clear
set more off
#delimit;
tempfile data;


local LI_name `"Insurance company"';
local BHC_name `"Bank"';
local LI_cluster;
local LI_cluster `"cluster(date)"';
local BHC_cluster `"cluster(date)"';
local measure `"bond"';
local idvar company_symbol;

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
qui unzipfile "$DataPath/trace_`sector'", replace;
use trace_`sector', clear;
erase trace_`sector'.dta;
/*Merge with cusip information from CUSIPM file*/
qui gen cusip8 = substr(cusip_id,1,8);
qui unzipfile "$DataPath/cusipm_`sector'", replace;
merge m:1 cusip8 using cusipm_`sector';
assert _m==3 | company_symbol=="AIG" | date>td(31mar2013);
keep if _m==3 | date>td(31mar2013); 
qui drop _m;
erase cusipm_`sector'.dta;

if `"`sector'"'==`"BHC"' {;
	qui drop if inlist(company_symbol,`"MET"',`"PFG"');
};

keep if inlist(trc_st,"G","T"); /*Removes corrected or cancelled trades*/
keep if inlist(sale_cndtn_cd,"@") | missing(sale_cndtn_cd); /*Keeps regular trades*/
keep if missing(asof_cd); /*Removes trade records reported with delay or corrections*/
drop if regexm(issue_desc_1,"[Ss][Uu][Bb]"); /*Drops subordinate notes*/
if `"`yield'"'==`"yield"' {; /*Drop trades with missing yield. Yield is an optional field*/
	keep if !missing(yld_pt);
};
qui gen double datetime = Clock(string(trd_exctn_dt,"%td")+trd_exctn_tm,`"DMYhms"');
qui format datetime %tC;
duplicates drop datetime cusip_id company_sym rptd_pr ascii_rptd_vol_tx, force; 
/*TRACE receives data from all FINRA parties to a transaction. Hence if the buyer and seller are both FINRA parties,
TRACE will have two observations for the same transaction*/
rename trd_exctn_dt date;
drop if maturity_date-date<730; /*Crude sample filter: drop bonds with less than 2 year remaining to maturity*/

qui gen QE = "";
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		if `"`alldates'"'==`""' & inlist(`"${QE`q'_`i'}"',`"25nov2008 8:15:00"',`"12aug2009 14:16:10"',`"4nov2009 14:19:00"',`"27aug2010 10:00:00"',`"3nov2010 14:16:39"',`"31aug2012 10:00:00"',`"12dec2012 12:30:02"',`"18dec2013 14:00:00"') {; 
			local dropdatelist `"`dropdatelist'`=dofC(`=tC(${QE`q'_`i'})')',"';
			local i = `i'+1;
			continue; /*Dates without any "action" in the Treasury market*/
		};
		else {;
			qui replace QE = `"QE`q'_`i'"' if date==dofC(`=tC(${QE`q'_`i'})');
			local datelist `"`datelist' QE`q'_`i'"';
			local i = `i'+1;
		};
	};
};

/*Identify most liquid issue on each date, by company*/
qui egen count = count(yld_pt), by(bond_sym_id date);
qui egen maxcount = max(count), by(company_symbol date);
qui gen mostliquid = count==maxcount;

/*Identify trades in event window*/
eventtime datetime, lower(`lower') upper(`upper') band(0:04:59) gen(bondeventtime);

/*Mean of trades in event window*/
foreach var of varlist yld_pt rptd_pr {;
	qui egen window_`var' = mean(`var') if !missing(bondeventtime), by(bond_sym_id date bondeventtime);
};

/*Keep trade closest to event window for each cusip*/
qui gen eventtime = .;
foreach date of local datelist {;
	qui egen double maxdatetime = max(datetime) if date==dofC(tC(${`date'})) & datetime<(tC(${`date'})-msofminutes(2)), by(cusip_id);
	qui egen double mindatetime = min(datetime) if date==dofC(tC(${`date'})) & datetime>(tC(${`date'})+msofminutes(18)), by(cusip_id);
	qui replace eventtime = -1 if datetime == maxdatetime;
	qui replace eventtime = 1 if datetime == mindatetime;
	qui drop maxdatetime mindatetime;
};
qui keep if inlist(eventtime,-1,1);
qui egen maxseq = max(msg_seq_nb), by(cusip_id date eventtime);
qui egen minseq = min(msg_seq_nb), by(cusip_id date eventtime);
drop if eventtime == -1 & msg_seq_nb!=maxseq;
drop if eventtime ==  1 & msg_seq_nb!=minseq;
rename datetime bonddatetime;

/*Identify cusip with narrowest window for each company and cusips in narrow window around the event*/
qui egen cusipdate = group(cusip_id date);
qui tsset cusipdate eventtime, delta(1);
qui gen windowsize = bonddatetime - L2.bonddatetime if eventtime==1;
qui egen windowsize_rank = rank(windowsize), by(company_symbol date) track;
qui gen inbondwindow = bondeventtime==1 & L2.bondeventtime==-1 if eventtime==1;

/*Merge yield to maturity of on the run 5 year Treasury*/
preserve;
qui use `"$DataPath/Treasuries-event-study-on-the-run"' if sectype==`sectype', replace;
eventtime datetime, lower(`lower') upper(`upper') band(0:04:59) gen(eventtime);
qui replace eventtime =  1 if tin(10jul2013 17:25,10jul2013 17:29:00) & `upper'>40;
qui collapse (mean) ytm datetime, by(date eventtime);
qui save `data'; 
restore;
merge m:1 date eventtime using `data', nogenerate assert(matched using) keep(matched);
drop if inlist(date,`dropdatelist'.);
qui drop if eventtime==0;

/*Merge data from CRSP daily stock file*/
preserve;
dsf `sector'; /*See program at end of file*/
tempfile dsf;
qui save `dsf', replace;
restore;
qui gen ticker = company_symbol;
merge m:1 date ticker using `dsf', nogenerate keep(master matched);

/*Regression variables*/
qui egen iddate = group(bond_sym_id date);
qui tsset cusipdate eventtime, delta(1);
qui gen logprice = log(rptd_pr);
qui gen logwindow_price = log(window_rptd_pr);
foreach var of varlist logprice logwindow_price ytm yld_pt window_yld_pt {;
	qui gen D`var' = S2.`var';
};
qui gen double DytmBP = 10^4 * Dytm; /*Converts units to basis points*/
qui gen double DytmBP10 = 10^3 * Dytm; /*Converts units to 10 basis points*/
label variable DytmBP10 `"10 basis point change in `sectype' year Treasury"';
foreach var of varlist Dyld_pt Dwindow_yld_pt Dlogprice Dlogwindow_price {;
	qui gen double `var'100 = 100*`var'; /*Variable in basis points*/
};
qui gen `sector' = !inlist(cusip_id,"78387GAQ6","61746SBC2","172967BF7","345397UC5","172967BH3","370425RZ5","13134VAA1","521865AE5","247126AD7") & !inlist(cusip_id,"345370CA6");
label variable `sector' `"``sector'_name'"';
qui drop if company_symbol=="AIG";

/*Generate weights. The weights are such that each company gets equal weight on each date regardless of the number of bonds outstanding.*/
qui levelsof date, local(dates);
qui levelsof company_symbol, local(tickers);
qui gen wgt = .;
foreach date of local dates {;
	foreach ticker of local tickers {;
		sum Dlogprice if date==`date' & company_symbol==`"`ticker'"', meanonly;
		qui replace wgt = 1/r(N) if date==`date' & company_symbol==`"`ticker'"';
	};
};		
	
/*************************************************************************************************************
2. Regressions
*************************************************************************************************************/
local n = 0;
foreach rhs in `"DytmBP10"' `"DytmBP10 c.`sector'#c.DytmBP10"' {;
	foreach cond in `""' `"if substr(QE,3,1)!="1""' {;
		qui eststo: reg Dlogprice100 `rhs' `cond' [aw=wgt], robust cluster(date);
		local n = `n' + 1;	
		if `"`cond'"'==`""' {;
			local indicate `"`indicate' & \multicolumn{1}{c}{Yes}"';
		};
		else {;
			local indicate `"`indicate' & \multicolumn{1}{c}{No}"';
		};		
	};
};

local indicate `"`"Includes QE1 `indicate' \\"'"';
local varlabels `"c.`sector'#c.DytmBP10 `"10 b.p. change X ``sector'_name'"'"';		
local groups `"& \multicolumn{`n'}{c}{Dependent variable: log change in bond price, p.p.}"'; 
local midrules `"\cmidrule(l{.75em}){2-`=`n'+1'}"';
local groups `"" `groups'\\ `midrules' ""';	
if e(N_clust)!=. {;
	local stats "r2 N_clust N";
	local stats_fmt "%9.3f %9.0f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Clusters"' `"Observations"' "';
};
else {;
	local stats "r2 N";
	local stats_fmt "%9.3f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Observations"' "';
};
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 6.6;
local title "``sector'_name' bond price response";
local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
local notes `"Notes: The dependent variable is the log change in the `measure' price over the shortest window covering at least 2 minutes before to 18 minutes after the announcement. The variable change in `sectype' year Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury from 2 minutes before to 18 minutes after the announcement. Standard errors in parentheses and clustered by date."';
local filename `"`sector'-`measure'-response-window-`lower'-`upper'`OUTPUT_PS'"';
local filenames `"`filenames' `filename'"';
esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(a2)) se(par fmt(a2) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
EmbedTex `"`filename'"' using `"$OutputPath/`filename'-embedded.tex"', title(`measure' price response) author(Gabriel Chodorow-Reich) replace;

/*Reduce to one observation per company*/
preserve;
if `"`yield'"'==`""' {;
	local depvar Dlogprice;
};
else {;
	local depvar Dyld_pt;
};
local window_depvar = `"Dwindow_"' + regexr(`"`depvar'"',`"^D"',`""');
foreach ps in "" 100 {;
	qui replace `depvar'`ps' = `window_depvar'`ps' if !missing(`window_depvar');
	qui egen mean`depvar'`ps' = mean(`depvar'`ps') if !missing(bondeventtime), by(company_symbol date bondeventtime);
	qui replace `depvar'`ps' = mean`depvar'`ps' if !missing(mean`depvar'`ps');
};
qui keep if windowsize_rank == 1; /*Keeps cusip with smallest window size for each company-date*/

/*Mean of market cap*/
qui egen marketcapbar = mean(marketcap), by(`idvar');

/*Add combined dates for median, mean, value-weighted mean*/
qui levelsof `idvar', local(`idvar's);
foreach pair in `"16dec2008 18mar2009"' `"22may2013 19jun2013"' `"10jul2013 18sep2013"' {;
	local d = `d' + 1;
	tokenize `pair';
	foreach `idvar' of local `idvar's {;
		qui set obs `=`=_N'+1';
		qui replace date = `d' in `=_N';
		foreach var of varlist `depvar' DytmBP {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(sum) if r(N)!=0 in `=_N';
		};
		qui replace DytmBP = . in `=_N' if r(N)!=2;
		foreach var of varlist marketcap* {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(mean) in `=_N';
		};
		qui replace `idvar' = `"``idvar''"' in `=_N';
	};
};

/*Add median, mean, value-weighted mean, and p-values of 2-sided t-test on each date to data set*/
qui levelsof date, local(dates);
qui gen Pvalue = .;
qui gen windowweight = 1/windowsize;
foreach date of local dates {;
	foreach stat in Median Mean vwMean {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace company_symbol = `"`stat'"' in `=_N';
		sum DytmBP if date == `date', meanonly;
		qui replace DytmBP = r(mean) in `=_N';
		if `"`stat'"'==`"Median"' {; 
			qui sum `depvar' if date == `date' & `sector' & !inlist(company_symbol,`"Median"',`"Mean"',`"vwMean"'), detail;
			qui replace `depvar' = r(p50) in `=_N';
		};		
		else {;
			if `"`stat'"'==`"Mean"' {;
				local wgt `""';
			};
			else if `"`stat'"'==`"vwMean"' {;
				local wgt `"[aw=marketcapbar]"';
			};
			foreach vce in "" ", robust" {;
				capture qui reg `depvar' if date==`date' & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"') `wgt' `vce';
				if _rc==0 {;
					qui replace `depvar' = _b[_cons] in `=_N';
					qui test _cons;
					/*Use maximum of conventional and robust SE*/
					qui replace Pvalue = r(p) in `=_N' if (r(p)>Pvalue[`=_N'] & !missing(r(p))) | missing(Pvalue[`=_N']);
				};
			};
		};
		qui replace `depvar'100 = 100*`depvar' in `=_N';
		qui replace eventtime = 1 in `=_N';
	};
};	

/*Drop combined dates with individual tickers*/
qui drop if inrange(date,1,3) & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"');

/*Drop combined dates with summary measures and add back later*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		foreach var of varlist `depvar' `depvar'100 DytmBP Pvalue {;
			sum `var' if `idvar'==`"`stat'"' & date==`date', meanonly;
			local `stat'`var'`date' = r(mean);
		};
	};
};
qui drop if inrange(date,1,3);

/*Mark outliers for readability of graph*/
qui gen outlier = 0;
if `"`sector'"'==`"BHC"' | `"`yield'"'==`"yield"' {;
	if `"`yield'"'!=`"yield"' {;
		local cutoff 0.06;
	};
	else {;
		local cutoff 1.5;
	};
	qui gen n = _n;
	qui levelsof date if abs(`depvar')>`cutoff' & `sector', local(dateoutliers);
	foreach date of local dateoutliers {;
		local dropdate;
		local datestring = string(`date',`"%tdnn/dd/CCYY"');
		qui levelsof n if abs(`depvar')>`cutoff' & date==`date' & `sector', local(outliers);
		foreach outlier of local outliers {;
			sum `depvar'100 in `outlier', meanonly;
			if r(mean)>0 {;
				local change `"+`=round(r(mean),1)'"';
			};
			else {;
				local change = round(r(mean),1);
			};
			local dropdate `"`dropdate'`=company_symbol[`outlier']'[`change'], "';
			qui replace n = 0 in `outlier';
		};
		local dropdate = `"`datestring' ("' + `"`=regexr(`"`dropdate'"',`", $"',`""')'"' + `")"';
		local dropdates `"`dropdates', `dropdate'"';
	};
	local dropdates = regexr(`"`dropdates'"',`"^, "',`""');
	local dropdates `" To enhance readability, the figure omits: `dropdates'."';
	qui replace outlier = 1 if n==0;
};


/*Scatterplot with dates labeled*/
if `"`yield'"'==`""' {;
	local depvar_desc `"Change in `measure' price, log points"';
	local depvar_short `"price"';
	local legend_pos 7;
};
else {;
	local depvar_desc `"Change in `measure' yield, basis points"';
	local depvar_short `"yield"';
	local legend_pos 11;
	local ylabels `"-150(50)150"';
};
local labsize medsmall;
local altangle1 75; /*Put some date labels at an alternate label to enhance readability*/
local altangle2 105; /*Put some date labels at an alternate label to enhance readability*/
qui levelsof DytmBP, local(values);foreach value of local values {;
	sum date if round(DytmBP,0.0001)==round(`value',0.0001), meanonly;
	if inlist(r(mean),`=td(9aug2011)',`=td(1dec2008)',`=td(13sep2012)') {;
		local tlabels`altangle1' `"`tlabels`altangle1'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else if inlist(r(mean),`=td(10aug2010)',`=td(18sep2013)',`=td(23sep2009)',`=td(22may2013)') {;
		local tlabels`altangle2' `"`tlabels`altangle2'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else {;
		local tlabels `"`tlabels' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
};
local xlabsize small; 
local xlabels `"xlabel(`tlabels', tposition(outside) labsize(`xlabsize') angle(vertical))"';
foreach a in `altangle1' `altangle2' {;
	local xlabels `"`xlabels' xlabel(`tlabels`a'', tposition(outside) labsize(`xlabsize') angle(`a') custom add)"';
};
if `"`sector'"'==`"LI"' {;
	local colorj = 0; local symbolj = 0; local j = 0; local scatters; local legendkey;
	local colors `"blue red green black brown purple brown cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
	local symbols `"O D T S + X"';
	qui levelsof company_symbol if !inlist(company_symbol,`"Median"',`"Mean"',`"vwMean"') & `sector', local(tickers);
	foreach ticker of local tickers {;
		local j = `j' + 1;
		foreach item in color symbol {;
			local `item'j = ``item'j' + 1;
			local `item': word ``item'j' of ``item's';
			if `"`item'"'==`""' {;
				local `item'j = 1;
				local `item': word ``item'j' of ``item's';
			};
		};
		local scatters `"`scatters' (scatter `depvar'100 DytmBP if company_symbol==`"`ticker'"' & !outlier, msymbol(`symbol') msize(small) mcolor(`color') xaxis(1 2))"';
		local legendkey `"`legendkey' label(`j' `"`ticker'"')"';
	};
	local j = `j'+1;
	local legend `"on col(3) rowgap(0) size(`labsize') position(`legend_pos') region(lcolor(white)) ring(0) `legendkey'"';
};
else {;
	local scatters `"(scatter `depvar'100 DytmBP if `sector' & !inlist(company_symbol,`"Median"',`"Mean"',`"vwMean"') & !outlier, msymbol(x) msize(small) mcolor(blue) xaxis(1 2))"';
	local scatters `"`scatters' (scatter `depvar'100 DytmBP if company_symbol==`"Median"', msymbol(O) msize(medium) mcolor(black) xaxis(1 2))"';
	local legend `"on col(1) rowgap(0) size(`labsize') position(11) region(lcolor(white)) ring(0) order(2) label(2 "Median")"';
};

twoway `scatters' if eventtime==1, yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(`ylabels',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot-window-`lower'-`upper'`alldates'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
local fullnote `"Notes: The change in bond `depvar_short' is over the shortest window containing at least `lower' minutes before to `upper' minutes after the announcement. The change in the on-the-run `sectype' year Treasury is the change in the yield to maturity from 2 minutes before to 18 minutes after the announcement.`dropdates'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(fig:`label') note(`fullnote') position(!p) replace;

/*Presentation figure*/
local labsize medlarge;
local xlabsize medium; 
local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',1);
local xlabels = subinstr(`"`xlabels'"',`"small"',`"`xlabsize'"',.);
twoway `scatters' if eventtime==1, yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(`ylabels',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response"';
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;


/*Add back combined dates with summary measures*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace `idvar' = `"`stat'"' in `=_N';
		foreach var of varlist `depvar' `depvar'100 DytmBP Pvalue {;
			qui replace `var' = ``stat'`var'`date'' in `=_N';
		};
	};
};

qui save "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'`alldates'", replace;

restore;

end;



/*************************************************************************************************************
3. Called programs
*************************************************************************************************************/
capture program drop dsf;
program define dsf;
syntax name(name=sector id="Sector LI, BHC, or SP500");
qui unzipfile "$DataPath/crspq_`sector'_dsf", replace;
use crspq_`sector'_dsf, clear;
erase crspq_`sector'_dsf.dta;
qui tsset permno date;
qui replace prc = -prc if prc<0; /*Negative price indicates observation is midpoint between bid and ask*/
qui gen double D1logprice = log(prc) - log(L.prc) if cfacpr==L.cfacpr; /*1 day change*/
qui replace D1logprice = log(prc) - log(L3.prc) if missing(D1logprice) & dow(date)==1 & cfacpr==L3.cfacpr; /*If event date on Monday, use prior Friday for prior observation*/
qui gen double marketcap = prc*shrout;
#delimit;
foreach pair in "2jan2008 24nov2008" {;
	tokenize `pair';
	forvalues s = 1/2 {;
		qui egen adjprc_``s'' = max(prc/cfacpr * (date==td(``s''))), by(permno);
	};
	qui gen double Dlogprice_`1'_`2' = log(adjprc_`2'/adjprc_`1');
};
tempfile dsf;
save `dsf';
qui unzipfile "$DataPath/CRSPQ_`sector'_stocknames", replace;
qui use crspq_`sector'_stocknames;
erase crspq_`sector'_stocknames.dta;
qui egen maxdate = max(nameenddt), by(permno);
qui keep if nameenddt==maxdate;
qui drop maxdate;
merge 1:m permno using `dsf', nogenerate keep(matched);
qui drop if nameenddt<date;
qui drop if inlist(comnam,"CANNON EXPRESS INC","INVERNESS MEDICAL TECHNOLOGY IN");
qui egen maxout = max(shrout), by(permco date);
qui drop if shrout!=maxout;
qui drop maxout;
end;

