clear all
set more off
#delimit;
tempfile data;
tempfile outdata;

/*************************************************************************************************************
This do-file constructs QE announcement window yield changes of Treasury yields.
*************************************************************************************************************/
do Paths;

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
do Treasuries-load-data;
do QE_dates;
global dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		local datelist `"`datelist' QE`q'_`i'"';
		local i = `i'+1;
	};
}; 

/*************************************************************************************************************
2. Estimate windows
*************************************************************************************************************/
#delimit;
rename datetime datetimesec;
qui gen double datetime = Cdhms(date,hhC(datetimesec),mmC(datetimesec),0); 
qui format datetime %tCDDmonCCYY_HH:MM; /*Will use to collapse into minute by minute instead of tick by tick*/
tempfile data;
qui save `data', replace;

*foreach date of local datelist {;
foreach date in QE2_4 {;

	di `"${`date'}"';

	qui use `data', clear;
	clear mata;
	mata mata clear;
	local datetime = Clock(`"${`date'}"',`"DMYhms"');
	qui keep if date == dofC(`datetime');
	
	/*Reduce dataset to the on the run securities on the date*/
	qui levelsof sectype, local(sectypes);
	foreach sectype of local sectypes {;
		 qui sum matdate if sectype == `sectype';
		 qui drop if matdate != r(max) & sectype == `sectype';
	};
			
	qui collapse (mean) indbid indask indbyld indayld, by(cusip kycrspid coupon date datetime matdate tdatdt tmatdt sectype *coupondate); 
	qui egen flatprice = rowmean(indbid indask);
	qui drop if missing(flatprice);

	/*Generate dirty (full) price. GovPx quotes clean (flat) prices, which exclude the value of accrued interest toward the next
	coupon payment. Yield curve fitting assumtes the dirty price. As best I can tell, the market convention is that the difference
	between the two is the coupon payment multiplied by the fraction of the coupon period gone by, i.e. not discounted.*/
	qui gen fullprice = flatprice + (coupon/2) * (date - lastcoupondate)/(nextcoupondate - lastcoupondate);
		
	/*Create yield to maturity and Macaulay duration for each security*/
	ytm fullprice coupon date matdate nextcoupondate, gen(ytm); /*GCR written program to compute yield to maturity*/
	Macaulay fullprice coupon date matdate ytm, gen(Macaulay); /*GCR written program to compute Macaulay duration*/
		
	/*Merge data from CRSP daily file*/
	if `"`date'"'!=`"QE3_7"' {; /*Missing CRSP data as of 2-19-2014*/
		qui gen caldt = date;
		qui unzipfile `"$DataPath/CRSPQ_UST_DLY"', replace;
		merge m:1 kycrspid caldt using CRSPQ_UST_DLY, nogenerate assert(matched using) keep(matched) nolabel nonotes noreport;
		qui gen Macaulay_CRSP = tdduratn/365 if tdduratn!=-99;
		move Macaulay_CRSP Macaulay;
		erase CRSPQ_UST_DLY.dta;
	};
		
	/*Append and save*/
	capture qui append using `outdata';
	qui save `outdata', replace;
};x;


/*************************************************************************************************************
3. Save data
*************************************************************************************************************/
use `outdata', clear;
qui egen id = group(cusip);
qui tsset id datetime;
qui save `"$DataPath/Treasuries-event-study-on-the-run"', replace;

