clear all
set more off
#delimit;
set matsize 11000;

/*************************************************************************************************************
This do-file does the MMF analysis at a weekly level. The subheadings 7e and 7f perform the analysis reported 
in the paper.

*************************************************************************************************************/
do Paths;
capture log close;
log using "$LogPath/MMF-weekly.log", replace;
tempfile data;
local sector MMF;
local sector_name `"Money market fund"';

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
do MMF-analyzer-clean;

/*************************************************************************************************************
2. Expenses regressions
*************************************************************************************************************/
#delimit;
capture drop idyear;
local sector MMF;
local sector_name `"Money market fund"';
qui egen idyear = group(id year);
foreach rhs in "i.year#c._7GDCYwkly" "i.year#c._7GDCYwkly i.year#c.IncdExpenseRatiomo" {;
	foreach FE in "daten" "idyear" {;
		qui eststo: areg ChgdExpenseRatiomowk `rhs', absorb(`FE') cluster(id_complex); 
		local nregs = `nregs' + 1;
		if `"`FE'"' == `"daten"' {;
			local daten_indicate `"`daten_indicate' & \multicolumn{1}{c}{Yes}"';
			local idyear_indicate `"`idyear_indicate' & \multicolumn{1}{c}{No}"';
		};
		else if `"`FE'"'==`"idyear"' {;
			local daten_indicate `"`daten_indicate' & \multicolumn{1}{c}{No}"';
			local idyear_indicate `"`idyear_indicate' & \multicolumn{1}{c}{Yes}"';
		};
		local bnames: colnames e(b);
		foreach bname of local bnames {; 
			if regexm(`"`bname'"',`"^[1-2][0-9][0-9][0-9][b]?[n]?\.year#c\._7GDCYwkly"') {; *Interaction coefficients; 
				local yyyy = substr(`"`bname'"',1,4);
				local varlabels `"`varlabels' `bname' `"Gross yield X `yyyy'"'"';
			};
			else if regexm(`"`bname'"',`"^[1-2][0-9][0-9][0-9][b]?[n]?\.year#c\.IncdExpenseRatiomo"') {; *Interaction coefficients;
				local yyyy = substr(`"`bname'"',1,4);
				local varlabels `"`varlabels' `bname' `"Incurred expenses X `yyyy'"'"';
			};
		};
	};
};

local daten_indicate `"Weekly FE `daten_indicate'\\"';
local idyear_indicate `"Year X fund share class FE `idyear_indicate'\\"'; 
local indicate `""`daten_indicate'" "`idyear_indicate'""';
local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable: weekly charged expenses}"'; 
local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
local groups `" "`groups1'\\ `midrules1'" "';	
local stats "N_clust N";
local stats_fmt "%12.0fc %12.0fc";
local stats_label `" `"Fund sponsor clusters"' `"Observations"' "';
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 5.5;
local title "`sector_name' determination of expenses charged";
local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
local notes `"Notes: The dependent variable is the weekly charged expense ratio. Standard errors in parentheses and clustered by fund sponsor."';
local filename `"`sector'-expense-regressions`OUTPUT_PS'"';
esttab * using "$PaperPath/`filename'.tex",  replace cells(b(star fmt(%9.2f)) se(par fmt(%9.2f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
estimates drop _all;

/*************************************************************************************************************
3. Expenses loadings
*************************************************************************************************************/
#delimit;
local sector MMF;
local rhs "i.daten#c._7GDCYwkly i.daten#c.IncdExpenseRatiomo";
local FE daten;
qui _pctile IncdExpenseRatiomo, p(0.01 99.99);
local l = r(r1);
local u = r(r2);
*local controls `"TaxFree Prime AnyInstitutional"';
/*areg ChgdExpenseRatiomowk `rhs' `controls' if inrange(IncdExpenseRatiomo,`l',`u'), absorb(`FE') cluster(id_complex);
estimates save `sector'_expenses_winsorized, replace;*/
local sector MMF;
local sector_name "Money market fund";
estimates use `sector'_expenses_winsorized.ster;
local bnames: colnames e(b);
foreach bname of local bnames {;
	if regexm(`"`bname'"',`"^[1-2][0-9][0-9][0-9][0-9][b]?[n]?\.daten#c\."') {; *Yield coefficients; 
		local j = `j' + 1;
		local rhs`j' = regexr(`"`bname'"',`".*#c\."',`""');
		local daten`j' = substr(`"`bname'"',1,5);
		foreach stat in b se {;
			local `stat'`j' = _`stat'[`bname'];
		};
	};
};

preserve;
clear;
qui gen rhs = ""; 
foreach var in daten b se {;
	qui gen `var' = .;
};
forvalues i = 1/`j' {;
	qui set obs `=`=_N'+1';
	qui replace rhs = `"`rhs`i''"' in `=_N';
	foreach var in daten b se {;
		qui replace `var' = ``var'`i'' in `=_N';
	};
};
qui format daten %td;
qui gen l = b + invnorm(0.025)*se;
qui gen u = b + invnorm(0.975)*se;
qui reshape wide b se l u, i(daten) j(rhs) string;
qui tsset daten;
foreach var of varlist b* {;
	qui tssmooth ma `var'_ma = `var', window(42 1 42);
};

local labsize medsmall;
local tfmt %tdMon-YY;

/*Figure of moving averages*/
local _7GDCY_lines `"(tsline b_7GDCYwkly_ma, lpattern(solid) lwidth(medthick) lcolor(blue))"';
local Incd_lines `"(tsline bIncdExpenseRatiomo , lpattern(solid) lwidth(medthick) lcolor(red))"';
local legend `"on row(2) rowgap(0) size(`labsize') position(3) region(lcolor(white)) ring(0) order(1 2) label(1 `"Gross yield"') label(2 `"Incurred expenses"') "';
twoway `_7GDCY_lines' `Incd_lines', subtitle("Percent of assets", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
local filename `sector'-expenses-regression-figure-2;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title = `"`sector_name' charged expenses determinants"';
local label `"fig:`filename'"';
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;

/*Paper figure*/
local lyear 2012;
foreach var in _7GDCYwkly IncdExpenseRatiomo {;
	sum b`var' if mofd(daten)==tm(`lyear'm1), meanonly;
	if regexm("`var'","_7GDCYwkly") {;
		local arrow_label `"Gross yield"';
		local color blue;
		local head = r(mean) - 0.07;
		local tail = r(mean) - 0.2;
		local clock_position 6;
	};
	else if regexm("`var'","IncdExpenseRatiomo") {;
		local arrow_label `"Incurred expenses"';
		local color red;
		local head = r(mean) + 0.07;
		local tail = r(mean) + 0.2;
		local clock_position 12;
	};
	local `var'_arrow `"(pcarrowi `tail' `=mdy(1,1,`lyear')' `head' `=mdy(1,1,`lyear')'  `""`arrow_label'""', msize(`labsize') barbsize(`labsize') mcolor(`color') lcolor(`color') mlabsize(`labsize') mlabcolor(`color') mlabpos(`clock_position'))"';	
	local arrows `"`arrows' ``var'_arrow'"';
	local `var'_lines `"(tsline b`var' l`var' u`var', lpattern(solid dot dot) lwidth(medthick thin thin) lcolor(`color' `color' `color'))"';
	local lines `"`lines' ``var'_lines'"';
}; 
local legend `"off"';
forvalues yyyy = 2006/2013 {;
	local tvalues `"`tvalues' `=mdy(1,1,`yyyy')'"';
};
twoway `lines' `arrows', subtitle("Loading", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
local filename `sector'-expenses-regression-figure;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title = `"Determinants of `=lower(`"`sector_name'"')' charged expenses"';
local label `"fig:`filename'"';
local fullnote `"Notes: The solid lines plot the weekly coefficients from an OLS regression of charged expenses on week fixed effects, gross yield, and incurred expenses. \Eqref{eq:MMF-expenses-regression} provides the estimating equation. The dotted lines plot 95\% confidence interval bands based on standard errors clustered by fund sponsor. The regression winsorizes observations with the smallest and largest 0.01 percent of incurred expenses."';
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;

/*Presentation figure*/
local labsize medlarge;
local arrows = subinstr(`"`arrows'"',`"medsmall"',`"`labsize'"',.);
local title = `"\large \\$\text{Charged}_{i,t} = \alpha_t + \gamma_{t}\left[ \text{Incurred}_{i,t}\right] + \beta_{t}\left[\text{yield}_{i,t}\right] + e_{i,t}$"';
twoway `lines' `arrows', subtitle("Loading", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;


restore;

/*The dip in Dec-12 is due to one fund with a jump in yield and no jump in expenses, but appears to be real data:
      |                            FundName          id   Chgd~k11   Chgd~k12   _7GDS~11   _7GDS~12 |
      |---------------------------------------------------------------------------------------------|
 305. | Dreyfus CA AMT-Free MuniCashMgt/Adm   26202K882        .23        .18        .23        .66 |
 306. | Dreyfus CA AMT-Free MuniCashMgt/Ins   26202K700        .22        .19        .22        .67 |
 307. | Dreyfus CA AMT-Free MuniCashMgt/Inv   26202K809        .22        .18        .22        .66 |
 308. | Dreyfus CA AMT-Free MuniCashMgt/Par   26202K601        .23        .19        .23        .67 |
*/
/*The dip in May-06 is due to one fund with a jump in incurred expenses and no jump in charged expenses, but it appears to be real data:
      |                             FundName          id   ChgdE~k4   ChgdE~k5   IncdEx~4   IncdEx~5 |
      |----------------------------------------------------------------------------------------------|
1144. |         Morgan Stanley ILF/Treas/Adm   61747C616         .2         .2       1.38       2.92 |
1145. |         Morgan Stanley ILF/Treas/Adv   61747C590         .3         .3       1.43       3.55 |
1146. |     Morgan Stanley ILF/Treas/CashMgt   61747C434        .15        .35       1.33       3.07 |
1147. |      Morgan Stanley ILF/Treas/InsSel   61747C558         .1         .1       1.28       2.82 |
1149. |         Morgan Stanley ILF/Treas/Inv   61747C574        .15        .15       1.33       2.86 |
      |----------------------------------------------------------------------------------------------|
1150. |        Morgan Stanley ILF/Treas/Part   61747C566        .55        .55       1.73       3.27 |
*/
 

/*************************************************************************************************************
4. Collapse to one observation per master fund-date
*************************************************************************************************************/
qui encode MasterClassFund, gen(MasterClassFund_n); /*1 if not master class, 2 if master class*/
qui egen HasMaster = max(MasterClassFund_n), by(id_fund daten);
qui egen tag = tag(id_fund daten) if HasMaster==1; 
qui keep if MasterClassFund=="True" | tag; /*Reduces data set to one observation per fund-date. I.e. keeps only one share class observation*/
qui drop tag;
duplicates drop id_fund daten, force; /*Corrects errors where multiple funds receive the master class designation*/ 
qui egen tag = tag(id_fund year);
qui xtset id_fund daten;

/*************************************************************************************************************
5. Yield by asset class regressions and figure
*************************************************************************************************************/
#delimit;
estimates drop _all;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;

preserve;
qui egen USTotal = rowtotal(USTreasury USOther);
qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);
assert USTotal + Repos + TimeDeposits + DomesticBankObligations + ForeignBankObligations + NonABCP_CommercialPaper + ABCP + FRNS == 100 | missing(_`grossyield',ABCP) if Prime;
local cats `"USOther Repos TimeDeposits DomesticBankObligations ForeignBankObligations FRNS NonABCP_CommercialPaper ABCP"';  
foreach cat of local cats {;
	qui replace `cat' = 10^(-4) * `cat'; /*So coefficients will have interpretation of basis points*/
	local cat_interactions `"`cat_interactions' ibn.daten#c.`cat'"';
};

/*Kaperczyk-Schnabl replication regressions*/
qui egen BankObligations = rowtotal(DomesticBankObligations ForeignBankObligations);
qui gen logsize = log(PortfolioAssetswkmils);
qui gen age = daten - date(InceptionDate,"MDY");
qui egen ComplexAssetswkmils = total(PortfolioAssetswkmils), by(id_complex);
qui gen logfamilysize = log(ComplexAssetswkmils);
local KScats = regexr(`"`cats'"',`"DomesticBankObligations ForeignBankObligations"',`"BankObligations"');
local KScats = regexr(`"`KScats'"',`"USOther"',`""');
foreach fundFE in `""' `"ib105.id_fund"' {;
	foreach period in `"1aug2007,31aug2008"' `"1jan2006,31jul2007"' {;
		qui eststo: areg Folio_`grossyield'E `KScats' logsize FolioChgdExpenseRatiomowk age logfamilysize `fundFE' if Prime & AnyInstitutional & tin(`period'), absorb(daten) cluster(id_fund);
	};
};
esttab * ,  cells(b(star fmt(a2)) se(par fmt(a3) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons *.daten *.id_fund, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
estimates drop _all;

local j;
forvalues yyyy = 2006/2013 {;

	qui reg Folio_`grossyield'E ibn.daten `cat_interactions' i.id_fund if year==`yyyy' & !TaxFree, noconstant;
	local bnames: colnames e(b);
	foreach bname of local bnames {;
		if regexm(`"`bname'"',`"^[1-2][0-9][0-9][0-9][0-9][b]?[n]?\.daten#c\."') {; 
			local j = `j' + 1; 
			local cat`j' = regexr(`"`bname'"',`".*#c\."',`""');
			local daten`j' = substr(`"`bname'"',1,5);
			foreach stat in b {;
				local `stat'`j' = _`stat'[`bname'];
			};
		};
	};	
};

clear;
qui gen cat = ""; 
foreach var in daten b {;
	qui gen `var' = .;
};
forvalues i = 1/`j' {;
	qui set obs `=`=_N'+1';
	qui replace cat = `"`cat`i''"' in `=_N'; 
	foreach var in daten b {;
		qui replace `var' = ``var'`i'' in `=_N';
	};
};
qui format daten %td;
qui gen year = year(daten);
table year cat, c(mean b);
qui reshape wide b, i(daten) j(cat) string;
qui tsset daten;

 
qui sum daten;
local maxdate = r(max);
local labsize medsmall;
local colors `"blue red green black brown purple cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
foreach cat of local cats {;
	rename b`cat' `cat';
	SplitName `cat', substitute(_ " ") label;
	local label: variable label `cat';
	qui sum `cat' if daten==`maxdate';
	local lastcat = r(mean);
	qui tssmooth ma `cat'_ma = `cat', window(42 1 42);
	local c = `c'+1;
	local lines `"`lines' (tsline `cat', lpattern(solid) lwidth(medthin) lcolor(`=word(`"`colors'"',`c')'))"';
	local lines_ma `"`lines_ma' (tsline `cat'_ma, lpattern(solid) lwidth(medthin) lcolor(`=word(`"`colors'"',`c')'))"';
	local addtext `"`addtext' ttext(`lastcat' `maxdate' `"`label'"', placement(e) size(`labsize') just(left) box bcolor(white) color(`=word(`"`colors'"',`c')'))"';
	local legendkey `"`legendkey' label(`c' `"`label'"')"';
};
local tfmt %tdMon-YY;
local legend `"on row(`c') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
forvalues yyyy = 2006/2013 {;
	local tvalues `"`tvalues' `=mdy(1,1,`yyyy')'"';
};
twoway `lines_ma' if tin(1jan2006,31mar2014), yline(0, lwidth(thin) lpattern(dash)) subtitle("Excess return over U.S. Treasuries, basis points", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
local filename `sector'-category-yield-figure`OUTPUT_PS';
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title = `"`sector_name' yields by category"';
local label `"fig:`filename'"';
local fullnote `"Notes: The figure plots the 3-month centered moving average of coefficients from a weekly regression of gross yield on the asset allocation in the categories shown. U.S. Treasuries are the omitted category, and the sample includes all taxable money market funds. The regression also includes a fund fixed effect allowed to vary by calendar year."';
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;

/*Presentation figure*/
local labsize medlarge;
local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',.);
twoway `lines_ma' if tin(1jan2006,31mar2014), yline(0, lwidth(thin) lpattern(dash)) subtitle("Excess return over U.S. Treasuries, basis points", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;


foreach cat of local cats {;
	rename `cat' `cat'_r;
};
tempfile cat_r;
qui save `cat_r';
restore;

merge m:1 daten using `cat_r', nogenerate;

/*************************************************************************************************************
6. Annual means and standard deviation of variables
*************************************************************************************************************/
#delimit;
foreach var of varlist Folio????ExpenseRatiomo Folio_7* Folio_1Mo* {;
	qui egen `var'_mu = mean(`var'), by(id_fund year);
	qui egen `var'_sd = sd(`var'), by(id_fund year);
};
foreach type in Chgd Incd {;
	foreach ps in `""' `"_mu"' {;
		qui gen Folio`type'Expense`ps' = Folio`type'ExpenseRatiomo`ps';
	};
};

foreach type in Chgd Incd {;
	forvalues yyyy = 2005/2013 {;
		qui gen `type'Expenses`yyyy'_1 = Folio`type'Expense_mu if year==`yyyy';
		qui egen `type'Expenses`yyyy' = mean(`type'Expenses`yyyy'_1), by(id_fund);
		qui drop `type'Expenses`yyyy'_1;
		qui gen `type'ExpensesX`yyyy' = Folio`type'Expense * (year==`yyyy');
		qui gen `type'Instrument`yyyy'_2005 = `type'Expenses2005 * (year==`yyyy');
		if `yyyy'>=2006 {;	
			qui gen `type'Instrument`yyyy'_2006 = `type'Expenses2006 * (year==`yyyy');
		};
	};
};
qui save `"$DataPath/MMF-analysis-weekly"', replace;

/*************************************************************************************************************
7. Reaching regressions, panel-IV
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"TaxFree Prime AnyInstitutional"';
local allcontrols; local endogenous; local instruments; local varlabels;
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

/*Form high minus low time-varying portfolio*/
qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	
qui gen HoldingRisk = ForeignBankObligations - Repos;
 
qui xi i.daten, prefix(_W);
qui xi i.year, prefix(_Y);

local spec panel;
local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in "" Prime {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
		
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;
	
		di `"`measure'"';

		preserve;
		qui egen count = count(id_fund) if year>=2006, by(id_fund);
		qui sum count;
		qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
		qui drop if tin(1sep2008,31dec2008); /*Drop last 4 months of 2008 due to unusual circumstances*/
	
		local controls `"`allcontrols'"';
			
		local condition; 
		local sample_note `" Regressions exclude observations between September 1 and December 31, 2008."';
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `" if Prime"';
			foreach control in TaxFree Prime {;	
				forvalues yyyy=2007/2013 {;
					local controls = regexr(`"`controls'"',`"`control'X`yyyy'"',`""');
				};
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `"`sample_note' Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};
		if `"`measure'"'==`"Folio_`grossreturn'_sd"' {;
			/*Replace 2008 measure with Jan-Aug*/
			qui egen sd08 = sd(Folio_`grossreturn') if year==2008, by(id_fund);
			qui replace `measure' = sd08 if year==2008;
			if `"`subset'"'==`"Prime"' {;
				local condition `"`condition' & tag"';
			};
			else {;
				local condition `"if tag"';
			};
			local WeekFE No;
			local YearFE Yes;
			local controls `"`controls' _Yyear*"';
		};
		else {;
			local WeekFE Yes;
			local YearFE No;
			local controls `"`controls' _Wdaten*"';
		};

		qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
		
		qui estadd local WeekFE `"`WeekFE'"';
		qui estadd local YearFE `"`YearFE'"';
		qui estadd local FundFE `"Yes"';
		qui estadd local SamplePeriod `"2006-13"';

		matrix first = e(first);
		matrix F = first["F",7];
		qui estadd scalar Fstat = F[1,1];
		qui estadd scalar KP = e(widstat);	
		local nregs = `nregs' + 1;
			
		restore;
	};
	
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "SamplePeriod FundFE WeekFE YearFE Fstat N_g N_clust N";
	local stats_fmt "%7s %3s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
	local stats_label `" `"Sample period"' `"Fund FE"' `"Week FE"' `"Year FE"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 4.6;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the weekly gross simple yield, annualized. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`sector'-reaching-`spec'-IV-regressions`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot( "\label{tab:`filename'}");	
	estimates drop _all;
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) replace;
qui copy "$OutputPath/`filename'1.tex" "$PaperPath/`filename'.tex", replace;

/*************************************************************************************************************
7b. Reaching regressions, panel-IV, split 2008-09 into three groups
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

/*Separate year for crisis period*/
qui replace year = 200809 if tin(1sep2008,30apr2009); 
qui drop *_mu *_sd ChgdExpenses* IncdExpenses* *Instrument* Folio????Expense Folio????Expense_mu;

foreach var of varlist Folio????ExpenseRatiomo Folio_7* Folio_1Mo* {;
	qui egen `var'_mu = mean(`var'), by(id_fund year);
	qui egen `var'_sd = sd(`var'), by(id_fund year);
};
foreach type in Chgd Incd {;
	foreach ps in `""' `"_mu"' {;
		qui gen Folio`type'Expense`ps' = Folio`type'ExpenseRatiomo`ps';
	};
};

foreach type in Chgd Incd {;
	foreach yyyy in 2005 2006 2007 2008 200809 2009 2010 2011 2012 2013 {;
		qui gen `type'Expenses`yyyy'_1 = Folio`type'Expense_mu if year==`yyyy';
		qui egen `type'Expenses`yyyy' = mean(`type'Expenses`yyyy'_1), by(id_fund);
		qui drop `type'Expenses`yyyy'_1;
		qui gen `type'ExpensesX`yyyy' = Folio`type'Expense * (year==`yyyy');
		qui gen `type'Instrument`yyyy'_2005 = `type'Expenses2005 * (year==`yyyy');
		if `yyyy'>=2006 {;	
			qui gen `type'Instrument`yyyy'_2006 = `type'Expenses2006 * (year==`yyyy');
		};
	};
};

local control_list `"TaxFree Prime AnyInstitutional"';
local allcontrols; local endogenous; local instruments; local varlabels;
foreach yyyy in 2007 2008 200809 2009 2010 2011 2012 2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

/*Form high minus low time-varying portfolio*/
qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	
qui gen HoldingRisk = ForeignBankObligations - Repos;
 
qui xi i.daten, prefix(_W);
qui xi i.year, prefix(_Y);

local spec panel;
local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in "" {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
		
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;
	
		di `"`measure'"';

		preserve;
		qui egen count = count(id_fund) if year>=2006, by(id_fund);
		qui sum count;
		qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
	
		local controls `"`allcontrols'"';
			
		local condition; 
		local sample_note `" 2008 corresponds to January 1 - August 31, 2008. 200809 corresponds to September 1, 2008 - April 30, 2009. 2009 corresponds to May 1 - December 31, 2009."';
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `" if Prime"';
			foreach control in TaxFree Prime {;	
				foreach yyyy in 2007 2008 200809 2009 2010 2011 2012 2013 {;
					local controls = regexr(`"`controls'"',`"`control'X`yyyy'"',`""');
				};
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `"`sample_note' Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};
		if `"`measure'"'==`"Folio_`grossreturn'_sd"' {;
			if `"`subset'"'==`"Prime"' {;
				local condition `"`condition' & tag"';
			};
			else {;
				local condition `"if tag"';
			};
			local controls = regexr(`"`controls'"',`"_Idaten\*"',`"_Iyear*"');
			local WeekFE No;
			local YearFE Yes;
			local controls `"`controls' _Yyear*"';
		};
		else {;
			local WeekFE Yes;
			local YearFE No;
			local controls `"`controls' _Wdaten*"';
		};

		qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
		
		qui estadd local WeekFE `"`WeekFE'"';
		qui estadd local YearFE `"`YearFE'"';
		qui estadd local FundFE `"Yes"';
		qui estadd local SamplePeriod `"2006-13"';

		matrix first = e(first);
		matrix F = first["F",7];
		qui estadd scalar Fstat = F[1,1];
		qui estadd scalar KP = e(widstat);	
		local nregs = `nregs' + 1;
			
		restore;
	};
	
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "SamplePeriod FundFE WeekFE YearFE Fstat N_g N_clust N";
	local stats_fmt "%7s %3s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
	local stats_label `" `"Sample period"' `"Fund FE"' `"Week FE"' `"Year FE"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 4.6;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the weekly gross simple yield, annualized. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`sector'-reaching-`spec'-IV-regressions-crisis-split`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot( "\label{tab:`filename'}");	
	estimates drop _all;
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;
*qui copy "$OutputPath/`filename'1.tex" "$PaperPath/`filename'.tex", replace;

/*Revert data set*/
qui use `"$DataPath/MMF-analysis-weekly"', clear;

/*************************************************************************************************************
7c. Reaching regressions, panel-IV, holdings only
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"AnyInstitutional"';
local allcontrols; local endogenous; local instruments; local varlabels;
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

qui xi i.daten, prefix(_W);
qui xi i.year, prefix(_Y);

local spec panel;
local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
local c; local filenames;

preserve;

qui keep if Prime;
qui egen count = count(id_fund) if year>=2006, by(id_fund);
qui sum count;
qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
qui drop if tin(1sep2008,31dec2008); /*Drop last 4 months of 2008 due to unusual circumstances*/

qui egen USTotal = rowtotal(USTreasury USOther);
qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);
assert USTotal + Repos + TimeDeposits + DomesticBankObligations + ForeignBankObligations + NonABCP_CommercialPaper + ABCP + FRNS == 100 | missing(_`grossyield',ABCP) if Prime;
local cats `"USTotal Repos TimeDeposits DomesticBankObligations ForeignBankObligations FRNS NonABCP_CommercialPaper ABCP"';  

local nregs; local weeks_indicate; local years_indicate;
local sample_note `" Regressions exclude observations between September 1 and December 31, 2008. Sample includes only Prime funds."';
local controls `"`allcontrols' _Wdaten*"';

foreach cat of local cats {;		

	di `"`cat'"';

	qui eststo: xtivreg2 `cat' `controls' (`endogenous' = `instruments'), fe cluster(id_complex) partial(`controls') first;			
		
	qui estadd local WeekFE `"Yes"';
	qui estadd local FundFE `"Yes"';
	qui estadd local SamplePeriod `"2006-13"';

	matrix first = e(first);
	matrix F = first["F",7];
	qui estadd scalar Fstat = F[1,1];
	qui estadd scalar KP = e(widstat);	
	local nregs = `nregs' + 1;
			
};
	
local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
local groups2 `"& \multicolumn{1}{L}{U.S.} & \multicolumn{1}{L}{Repo} & \multicolumn{1}{L}{Time} & \multicolumn{1}{L}{Domestic} & \multicolumn{1}{L}{Foreign}  & \multicolumn{1}{L}{FRNS} & \multicolumn{1}{L}{Non ABCP} & \multicolumn{1}{L}{ABCP}"'; 
local groups3 `"& \multicolumn{1}{L}{obligations} &  & \multicolumn{1}{L}{deposits} & \multicolumn{1}{L}{bank} & \multicolumn{1}{L}{bank}  &  & \multicolumn{1}{L}{CP} "'; 
local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
local stats "SamplePeriod FundFE WeekFE Fstat N_g N_clust N";
local stats_fmt "%7s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
local stats_label `" `"Sample period"' `"Fund FE"' `"Week FE"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 2.2;
local title "`sector_name' allocation regressions";
local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
local notes `"Notes: `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
local filename `"`sector'-allocation-IV-regressions`OUTPUT_PS'"';
esttab * using "$OutputPath/`filename'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
estimates drop _all;
EmbedTex `"`filename'.tex"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) landscape append;

/*************************************************************************************************************
7d. Reaching regressions, panel-IV, annual means
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"TaxFree Prime AnyInstitutional"';
local allcontrols; local endogenous; local instruments; local varlabels;
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

/*Form high minus low time-varying portfolio*/
qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	
qui gen HoldingRisk = ForeignBankObligations - Repos;
 
qui xi i.year, prefix(_Y);

local spec panel-annual;
local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in "" Prime {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
		
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;
	
		di `"`measure'"';

		preserve;
		if !inlist(`"`measure'"',`"Folio_`grossreturn'_sd"') {;
			qui egen mean`measure' = mean(`measure'), by(id_fund year);
			qui replace `measure' = mean`measure';
		};
		qui egen count = count(id_fund) if year>=2006, by(id_fund);
		qui sum count;
		qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
		qui drop if tin(1sep2008,31dec2008); /*Drop last 4 months of 2008 due to unusual circumstances*/
	
		local controls `"`allcontrols' _Yyear*"';
			
		local condition `"if tag"'; 
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `"`condition' & Prime"';
			foreach control in TaxFree Prime {;	
				forvalues yyyy=2007/2013 {;
					local controls = regexr(`"`controls'"',`"`control'X`yyyy'"',`""');
				};
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `" Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};

		qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
		
		qui estadd local YearFE `"Yes"';
		qui estadd local FundFE `"Yes"';
		qui estadd local SamplePeriod `"2006-13"';

		matrix first = e(first);
		matrix F = first["F",7];
		qui estadd scalar Fstat = F[1,1];
		qui estadd scalar KP = e(widstat);	
		local nregs = `nregs' + 1;
			
		restore;
	};
	
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "SamplePeriod FundFE YearFE Fstat N_g N_clust N";
	local stats_fmt "%7s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
	local stats_label `" `"Sample period"' `"Fund FE"' `"Year FE"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 4.6;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the annual average of the weekly gross simple yield. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`sector'-reaching-`spec'-IV-regressions`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot( "\label{tab:`filename'}");	
	estimates drop _all;
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;
*qui copy "$OutputPath/`filename'1.tex" "$PaperPath/`filename'.tex", replace;

/*************************************************************************************************************
7e. Reaching regressions, panel-IV, annual means, control for initial holdings
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
qui use "$DataPath/MMF-analysis-weekly", clear;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);

local cats `"USOther Repos TimeDeposits DomesticBankObligations ForeignBankObligations FRNS NonABCP_CommercialPaper ABCP"';  
foreach cat of local cats {;
	if length(`"`cat'"')>20 {;
		local catshort = substr(`"`cat'"',1,20);
		qui gen `catshort' = `cat';
	};
	else {;
		local catshort `cat';
	};
	qui egen `catshort'05a = mean(`cat') if year==2005, by(id_fund);
	qui egen `catshort'2005 = mean(`catshort'05a), by(id_fund);
	qui drop `catshort'05a;
	local cat_controls `"`cat_controls' `catshort'2005"';
};

local control_list `"TaxFree Prime AnyInstitutional `cat_controls'"';
local allcontrols; local endogenous; local instruments; local varlabels;
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

/*Form high minus low time-varying portfolio*/
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	
qui gen HoldingRisk = ForeignBankObligations - Repos;
*qui replace HoldingRisk = HoldingRiskTV;
 
qui xi i.year, prefix(_Y);

local spec panel-annual-cat-controls;
local controls_note `"The time-varying controls are categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo), 2005 portfolio share by asset class, and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in "" Prime {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
		
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;
	
		di `"`measure'"';

		preserve;
		if !inlist(`"`measure'"',`"Folio_`grossreturn'_sd"') {;
			qui egen mean`measure' = mean(`measure'), by(id_fund year);
			qui replace `measure' = mean`measure';
		};
		qui egen count = count(id_fund) if year>=2006, by(id_fund);
		qui sum count;
		qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
	
		local controls `"`allcontrols' _Yyear*"';
			
		local condition `"if tag"'; 
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `"`condition' & Prime"';
			foreach control in TaxFree Prime {;	
				forvalues yyyy=2007/2013 {;
					local controls = regexr(`"`controls'"',`"`control'X`yyyy'"',`""');
				};
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `" Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};
		if !inlist(`"`measure'"',`"Folio_`grossyield'"', `"Folio_`grossreturn'_sd"') {;
			local controls `"_Yyear*"';
		};

		qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
		
		qui estadd local YearFE `"Yes"';
		qui estadd local FundFE `"Yes"';
		if inlist(`"`measure'"',`"Folio_`grossyield'"', `"Folio_`grossreturn'_sd"') {;
			qui estadd local CatFE `"Yes"';
		};
		else {;
			qui estadd local CatFE `"No"';
		};
		qui estadd local SamplePeriod `"2006-13"';

		qui estadd scalar KP = e(widstat);	
		matrix first = e(first);
		matrix F = first["F",7];
		if F[1,1]!=. {;
			qui estadd scalar Fstat = F[1,1];
		};
		else {;		
			 qui estimates dir; /*Required so Fstat gets posted to main regression*/
			 local lastest: word `:word count `r(names)'' of `r(names)';
			 qui xtreg IncdExpensesX2013 `instruments' `controls' `condition', fe cluster(id_complex);
			 qui estadd scalar Fstat = `=(_b[IncdInstrument2013_2005]/_se[IncdInstrument2013_2005])^2': `lastest';
		};
		local nregs = `nregs' + 1;
			
		restore;
	};
	
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "SamplePeriod FundFE YearFE CatFE Fstat N_g N_clust N";
	local stats_fmt "%7s %3s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
	local stats_label `" `"Sample period"' `"Fund FE"' `"Year FE"' `"Time-varying controls"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 1.3;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[!t]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the annual average of the weekly gross simple yield. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`sector'-reaching-`spec'-IV-regressions`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx {1}{c} {1}{L} "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot( "\label{tab:`filename'}");	
	estimates drop _all;
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;
qui copy "$OutputPath/`filename'1.tex" "$PaperPath/`filename'.tex", replace;

/*************************************************************************************************************
7f. Reaching regressions, week by week, controlling for initial allocation
*************************************************************************************************************/
#delimit;
use "$DataPath/MMF-analysis-weekly", clear;
#delimit;
set more off;
estimates drop _all;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);

local cats `"USOther Repos TimeDeposits DomesticBankObligations ForeignBankObligations FRNS NonABCP_CommercialPaper ABCP"';  
foreach cat of local cats {;
	if length(`"`cat'"')>20 {;
		local catshort = substr(`"`cat'"',1,20);
		qui gen `catshort' = `cat';
	};
	else {;
		local catshort `cat';
	};
	qui egen `catshort'05a = mean(`cat') if year==2005, by(id_fund);
	qui egen `catshort'2005 = mean(`catshort'05a), by(id_fund);
	qui drop `catshort'05a;
	local cat_controls `"`cat_controls' `catshort'2005"';
};

local control_list `"TaxFree Prime AnyInstitutional `cat_controls'"';

/*Form high minus low time-varying portfolio*/
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	

qui egen count = count(id_fund), by(daten);
qui sum count;
qui drop if count<100;
qui drop count;
qui levelsof daten if year>=2007, local(weeks);

local spec weekly;
local controls_note `"Regressions also include categorical variables for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in ""  {;
		
	foreach measure in Folio_`grossyield' HoldingRisk  {;
		
		di `"`measure'"';

		foreach week of local weeks {;
				
			preserve;
			qui keep if (year==2006) | (daten==`week');
			qui drop if Folio_`grossyield'==0; /*Crude sample filter*/
			
			qui egen count = count(id_fund), by(id_fund);
			qui sum count;
			qui keep if count==r(max); /*Restricts sample to balanced panel*/

			qui xi i.daten, prefix(_W);
	
			local uninteracted_controls `"`control_list'"';
			local controls;
			
			local condition; 
			if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
				local condition `" if Prime"';
				local uninteracted_controls = regexr(`"`uninteracted_controls'"',`"`control'"',`""');
			};
		
			foreach control of local uninteracted_controls {;
				qui gen `control'Xweek = `control' * (daten==`week');
				local controls `"`controls' `control'Xweek"';
			};
			local controls `"`controls' _W*"';
			
			qui gen endogenous = FolioIncdExpense * (daten==`week');
			qui gen instrument = IncdExpenses2005 * (daten==`week');

			qui xtivreg2 `measure' `controls' (endogenous = instrument) `condition', fe cluster(id_complex) partial(`controls');			
		
			foreach stat in b se {;
				local `subset'`measure'_`stat'`week' = _`stat'[endogenous];
			};
			local `subset'`measure'_N`week' = e(N);
			
			restore;
		};
	};
};

clear;
qui gen subset = ""; qui gen measure = ""; qui gen daten = .; qui gen b = .; qui gen se = .; qui gen N = .;

foreach subset in ""  {;
	foreach measure in Folio_`grossyield' HoldingRisk  {;
		foreach week of local weeks {;
			qui set obs `=`=_N'+1';
			qui replace subset = `"`subset'"' in `=_N';
			qui replace measure = `"`measure'"' in `=_N';
			qui replace daten = `week' in `=_N';
			foreach stat in b se N {;
				qui replace `stat' = ``subset'`measure'_`stat'`week'' in `=_N';
			};
		};
	};
};

qui gen l = b + invnorm(0.025)*se;
qui gen u = b + invnorm(0.975)*se;
qui egen id = group(subset measure), missing;
qui tsset id daten;
local lines `"(tsline b l u, lwidth(medthick thin thin) lcolor(blue blue blue) lpattern(solid dash dash))"';

/*Figures of regression coefficients*/
local filenames;
local tfmt %tdMon-YY;
local legend `"off"';
forvalues yyyy = 2007/2013 {;
	local tvalues `"`tvalues' `=mdy(1,1,`yyyy')'"';
};
local yvalues "#6";

local Folio_`grossyield'_name `"gross yield"';
local HoldingRisk_name `"holding risk"';
local WAMwk_name `"duration risk"';

foreach subset in ""  {;
	if `"`subset'"'==`"Prime"' {;
		local Sample_note `" Sample excludes fund-weeks with a gross yield of zero and non-Prime funds"';
	};
	else {;
		local Sample_note `" Sample excludes fund-weeks with a gross yield of zero."';
	};

	foreach measure in Folio_`grossyield' HoldingRisk WAMwk {;

		/*Paper figure*/
		local labsize medsmall;
		twoway `lines' if subset==`"`subset'"' & measure==`"`measure'"', yline(0, lpattern(dash) lwidth(thin)) subtitle("Regression coefficient", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
		local filename `sector'-reaching-regression-figure-`=regexr(`"``measure'_name'"',`" "',`"-"')';
		qui graph export `"$OutputPath/MMF_figures/`filename'.eps"', replace;
		local title = `"Loading of ``measure'_name' on incurred expenses, by week"';
		local label `"fig:`filename'"';
		local fullnote `"Notes: The solid line plots the coefficients from a weekly regression of `=lower(`"``measure'_name'"')' on incurred expenses, with incurred expenses instrumented using their 2005 average value. The regression also contains fund fixed effects identified by including observations from 2006, and fund type and asset class holdings in 2005 interacted with week.`Sample_note' The dotted lines plot 95\% confidence interval bands based on standard errors clustered by fund sponsor."';
		TexFigure `"`filename'.eps"' using `"$OutputPath/MMF_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
		local filenames `"`filenames' `filename'.tex"';
		
		if `"`subset'"'==`""' & `"`measure'"'==`"Folio_`grossyield'"' {;
			qui copy `"$OutputPath/MMF_figures/`filename'.eps"' `"$PaperPath/`filename'.eps"', replace;
			TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(tb) replace;
		};
		
		/*Presentation figure*/
		local labsize medlarge;
		twoway `lines' if subset==`"`subset'"' & measure==`"`measure'"', yline(0, lpattern(dash) lwidth(thin)) subtitle("Regression coefficient", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(`yvalues',nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tvalues',tposition(outside) labsize(`labsize') format(`tfmt')) ysize($ysize_p) xsize($xsize_p);
		qui graph export `"$OutputPath/MMF_figures/Presentation-`filename'.eps"', replace;
		TexFigure `"input-files/`filename'.eps"' using `"$OutputPath/MMF_figures/Presentation-`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;
	};
};
EmbedTex `"`filenames'"' using `"$OutputPath/MMF_figures/`sector'-reaching-regression-figures.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) replace;
qui format daten %td;
qui save "$DataPath/`sector'-reaching-regression-figures-data", replace;

/*************************************************************************************************************
7g. Reaching regressions, scatterplots, selected dates
*************************************************************************************************************/
#delimit;
use "$DataPath/MMF-analysis-weekly", clear;
set more off;
estimates drop _all;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local measure Folio_`grossyield';
qui gen TypeOther = !Prime & !TaxFree;

local dates `"26jun2007 30jun2009 25jun2013 31dec2013"';

local labsize medlarge;
local colors `"blue red green"';
local symbols `"circle triangle square"';
foreach type in Prime TaxFree TypeOther {;
	local i = `i' + 1;
	local scatters `"`scatters' (scatter `measure' IncdExpenses2005 if `type', msymbol(`:word `i' of `symbols'') msize(vsmall) mcolor(`:word `i' of `colors'')) (lfit `measure' IncdExpenses2005 if `type', lcolor(`:word `i' of `colors'') lpattern(dash) lwidth(medthin))"';
};
local legend `"on row(3) rowgap(0) size(`labsize') position(5) region(lcolor(white)) ring(0) order(1 3 5) label(1 "Prime funds") label(3 "Tax free") label(5 "Other")"';

foreach date of local dates {;
		
	preserve;
	qui keep if daten==td(`date'); 
	qui drop if Folio_`grossyield'==0; /*Crude sample filter*/
	local ytitle `"Gross yield on assets, `=string(td(`date'),`"%tdnn/dd/CCYY"')'"';
					
	twoway `scatters', legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=$l_m_p r=$r_m_p)) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`ytitle'", size(`labsize')) xtitle("Incurred expenses as percent of assets, 2005", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize($ysize_p) xsize($xsize_p);
	local filename `sector'-reaching-scatter-`date';
	qui graph export `"$PresentationPath/`filename'.eps"', replace;
	local title `"Bivariate reduced form, `=string(td(`date'),`"%tdMonth_dd,_CCYY"')'"';
	TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(t) replace;
			
	restore;
};


/*************************************************************************************************************
7h. Reaching regressions, panel-IV, annual means, control for current holdings
*************************************************************************************************************/
#delimit;
set more off;
estimates drop _all;
qui use "$DataPath/MMF-analysis-weekly", clear;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

qui egen NonABCP_CommercialPaper = rowtotal(FirstTierCommercialPaper SecondTierCommercialPaper);
qui replace NonABCP_CommercialPaper = NonABCP_CommercialPaper - ABCP if !missing(ABCP);

local cats `"USOther Repos TimeDeposits DomesticBankObligations ForeignBankObligations FRNS NonABCP_CommercialPaper ABCP"';  
foreach cat of local cats {;
	if length(`"`cat'"')>20 {;
		local catshort = substr(`"`cat'"',1,20);
		qui gen `catshort' = `cat';
	};
	else {;
		local catshort `cat';
	};
	qui egen `catshort'yyyy = mean(`cat'), by(id_fund year);
	local cat_controls `"`cat_controls' `catshort'yyyy"';
};

local control_list `"TaxFree Prime AnyInstitutional `cat_controls'"';
local allcontrols; local endogenous; local instruments; local varlabels;
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV: 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
di `"`endogenous'"';
di `"`instruments'"';
di `"`allcontrols'"';

/*Form high minus low time-varying portfolio*/
qui describe *_ma, varlist;
foreach var of varlist `r(varlist)' {;
	local cat = regexr(`"`var'"',`"_ma$"',`""');
	local cats `"`cats' `cat'"';
	local newvarlist `"`newvarlist' `cat'_rank"';
};
qui rowranks *_ma, gen(`newvarlist'); /*Ranks variables within an observation*/
qui gen HoldingRiskTV = -USTreasury;
foreach cat of local cats {;
	qui replace HoldingRisk = HoldingRisk + `cat' if inlist(`cat'_rank,7,8);
};	
qui gen HoldingRisk = ForeignBankObligations - Repos;
*qui replace HoldingRisk = HoldingRiskTV;
 
qui xi i.year, prefix(_Y);

local spec panel-annual-current-cat-controls;
local controls_note `"The time-varying controls are categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo), current portfolio share by asset class, and for whether the fund has any institutional shares."'; 
local c; local filenames;

foreach subset in "" Prime {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
		
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd WAMwk {;
	
		di `"`measure'"';

		preserve;
		if !inlist(`"`measure'"',`"Folio_`grossreturn'_sd"') {;
			qui egen mean`measure' = mean(`measure'), by(id_fund year);
			qui replace `measure' = mean`measure';
		};
		qui egen count = count(id_fund) if year>=2006, by(id_fund);
		qui sum count;
		qui keep if count==r(max); /*Restricts sample to balanced panel over 2006-13*/
	
		local controls `"`allcontrols' _Yyear*"';
			
		local condition `"if tag"'; 
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `"`condition' & Prime"';
			foreach control in TaxFree Prime {;	
				forvalues yyyy=2007/2013 {;
					local controls = regexr(`"`controls'"',`"`control'X`yyyy'"',`""');
				};
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `" Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};
		if !inlist(`"`measure'"',`"Folio_`grossyield'"', `"Folio_`grossreturn'_sd"') {;
			local controls `"_Yyear*"';
		};

		qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
		
		qui estadd local YearFE `"Yes"';
		qui estadd local FundFE `"Yes"';
		if inlist(`"`measure'"',`"Folio_`grossyield'"', `"Folio_`grossreturn'_sd"') {;
			qui estadd local CatFE `"Yes"';
		};
		else {;
			qui estadd local CatFE `"No"';
		};
		qui estadd local SamplePeriod `"2006-13"';

		qui estadd scalar KP = e(widstat);	
		matrix first = e(first);
		matrix F = first["F",7];
		if F[1,1]!=. {;
			qui estadd scalar Fstat = F[1,1];
		};
		else {;		
			 qui estimates dir; /*Required so Fstat gets posted to main regression*/
			 local lastest: word `:word count `r(names)'' of `r(names)';
			 qui xtreg IncdExpensesX2013 `instruments' `controls' `condition', fe cluster(id_complex);
			 qui estadd scalar Fstat = `=(_b[IncdInstrument2013_2005]/_se[IncdInstrument2013_2005])^2': `lastest';
		};
		local nregs = `nregs' + 1;
			
		restore;
	};
	
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "SamplePeriod FundFE YearFE CatFE Fstat N_g N_clust N";
	local stats_fmt "%7s %3s %3s %3s %9.1f %12.0fc %12.0fc %12.0fc";
	local stats_label `" `"Sample period"' `"Fund FE"' `"Year FE"' `"Time-varying controls"' `"2013 first stage F statistic"' `"Unique funds"' `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 1.3;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[!t]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\\" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the annual average of the weekly gross simple yield. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor. +,*,** indicate significance at the 0.1, 0.05, 0.01 levels respectively."';
	local filename `"`sector'-reaching-`spec'-IV-regressions`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx {1}{c} {1}{L} "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot( "\label{tab:`filename'}");	
	estimates drop _all;
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;


x;
/*************************************************************************************************************
8. Reaching regressions, instrumenting
*************************************************************************************************************/
#delimit;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX* HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"TaxFree Prime AnyInstitutional"';
local varlabels;
local allcontrols;
forvalues yyyy = 2010/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV with 2006 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2006"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};
local allcontrols_note `"categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 

qui gen HoldingRisk = ForeignBankObligations - Repos;

local c;
foreach subset in "" Prime {;
	foreach spec in "" "lagged_depvar" "first_diff" {;
		local nregs; local weeks_indicate; local years_indicate;
		local c = `c' + 1;
	
		foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;

			preserve;

			local controls_note `"Regressions also include `allcontrols_note'"';
			local depvar_desc `"Dependent variable:"';

			if inlist(`"`spec'"',`"lagged_depvar"',`"first_diff"') {;
				qui gen `measure'2006 = `measure' if year==2006;
				qui egen lagged_depvar = mean(`measure'2006), by(id_fund);
				if inlist(`"`spec'"',`"lagged_depvar"') {;
					forvalues yyyy = 2010/2013 {;
						qui gen lagged_depvarX`yyyy' = lagged_depvar * (year==`yyyy');
						local varlabels `"`varlabels' lagged_depvarX`yyyy' `"2006 mean of dep. var. X year==`yyyy'"'"';
						local lagged_depvars`c' `"`lagged_depvars`c'' lagged_depvarX`yyyy'"';
					};
					local controls_note `"Regressions also include the 2006 value of the dependent variable, interacted with year, and `allcontrols_note'"';
				};
				else if inlist(`"`spec'"',`"first_diff"') {;
					qui replace `measure' = `measure' - lagged_depvar;
					local depvar_desc `"Dep. var.: difference from 2006 of"';
				};
			};

			qui keep if inrange(year,2010,2013);	
			local controls `"`allcontrols' _Idaten*"';
		
			local condition; local sample_note;
			if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
				local condition `" if Prime"';
				local controls = subinstr(`"`controls'"',`"*X*"',`""',.);
				if `"`subset'"'==`"Prime"' {;
					local sample_note `" Sample includes only Prime funds."';
					local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
				};
			};
			if `"`measure'"'==`"Folio_`grossreturn'_sd"' {;
				if `"`subset'"'==`"Prime"' {;
					local condition `"`condition' & tag"';
				};
				else {;
					local condition `"if tag"';
				};
				local controls = regexr(`"`controls'"',`"_Idaten\*"',`"_Iyear*"');
				local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{No}"';
				local years_indicate `"`years_indicate' & \multicolumn{1}{c}{Yes}"';
				qui xi i.year;
			};
			else {;
				local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{Yes}"';
				local years_indicate `"`years_indicate' & \multicolumn{1}{c}{No}"';
				qui xi i.daten;
			};
			
			qui eststo: ivreg2 `measure' `lagged_depvars`c'' `controls' (`endogenous' = `instruments') `condition', cluster(id_complex) partial(`lagged_depvars`c'' `controls') first;
			matrix first = e(first);
			matrix F = first["F",4];
			qui estadd scalar Fstat = F[1,1];
			qui estadd scalar KP = e(widstat);
			
			local nregs = `nregs' + 1;
			
			restore;
		};

		local weeks_indicate `""Week FE `weeks_indicate'\\""';	
		local years_indicate `""Year FE `years_indicate'\\""';			
		local indicate `" `weeks_indicate' `years_indicate' "';					
		local groups1 `"& \multicolumn{`nregs'}{c}{`depvar_desc'}"'; 
		local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
		local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
		local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
		local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
		local stats "Fstat N_clust N";
		local stats_fmt "%9.1f %12.0fc %12.0fc";
		local stats_label `" `"2013 first stage F statistic"' `"Fund sponsor clusters"' `"Observations"' "';
		local num_stats: word count `stats'; 
		local layout;
		forvalues l = 1/`num_stats' {;
			local layout `"`layout' "\multicolumn{1}{c}{@}" "';
		};
		local si 3.4;
		local title "`sector_name' reaching for yield regressions";
		local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
		local prehead `"prehead(`table_preamble' `groups')"';			
		local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
		local notes `"Notes: The gross yield is the weekly gross simple yield, expressed on an annualized basis. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Incurred expenses in each year are instrumented using the fund's 2006 incurred expenses. Standard errors in parentheses and clustered by fund sponsor."';
		local filename `"`sector'-reaching-regressions`OUTPUT_PS'"';
		local filenames `"`filenames' `filename'`c'.tex"';
		esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls', relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
		estimates drop _all;

	};
};
EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;



x;

/*************************************************************************************************************
9. Reaching regressions, DiD
*************************************************************************************************************/
#delimit;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"TaxFree Prime AnyInstitutional"';
local rhs `"IncdExpenses2006"';
local varlabels `"IncdExpenses2006 `"2006 incurred expenses"'"';
local allcontrols `"`control_list'"';
forvalues yyyy = 2010/2013 {;
	local rhs `"`rhs' IncdInstrument`yyyy'_2006"';
	local varlabels `"`varlabels' IncdInstrument`yyyy'_2006 `"2006 incurred expenses X year==`yyyy'"'"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};

local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 

qui gen HoldingRisk = ForeignBankObligations - Repos;

local c; local filenames;
foreach subset in "" Prime {;
	local nregs; local weeks_indicate; local years_indicate;
	local c = `c' + 1;
	
	foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;

		preserve;

		qui keep if year==2006 | inrange(year,2010,2013);	
		local controls `"`allcontrols' _Idaten*"';
		
		local condition; local sample_note;
		if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
			local condition `" if Prime"';
			foreach control in TaxFree Prime {;	
				local controls = regexr(`"`controls'"',`"`control'"',`""');
				local controls = regexr(`"`controls'"',`"(`control'X201[0123] )+"',`""');
			};
			if `"`subset'"'==`"Prime"' {;
				local sample_note `" Sample includes only Prime funds."';
				local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
			};
		};
		if `"`measure'"'==`"Folio_`grossreturn'_sd"' {;
			if `"`subset'"'==`"Prime"' {;
				local condition `"`condition' & tag"';
			};
			else {;
				local condition `"if tag"';
			};
			local controls = regexr(`"`controls'"',`"_Idaten\*"',`"_Iyear*"');
			local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{No}"';
			local years_indicate `"`years_indicate' & \multicolumn{1}{c}{Yes}"';
			qui xi i.year;
		};
		else {;
			local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{Yes}"';
			local years_indicate `"`years_indicate' & \multicolumn{1}{c}{No}"';
			qui xi i.daten;
		};
			
		qui eststo: ivreg2 `measure' `controls' `rhs' `condition', cluster(id_complex) partial(`controls');
	
		local nregs = `nregs' + 1;
		
		restore;
	};

	local weeks_indicate `""Week FE `weeks_indicate'\\""';	
	local years_indicate `""Year FE `years_indicate'\\""';			
	local indicate `" `weeks_indicate' `years_indicate' "';					
	local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
	local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
	local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
	local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
	local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
	local stats "N_clust N";
	local stats_fmt "%12.0fc %12.0fc";
	local stats_label `" `"Fund sponsor clusters"' `"Observations"' "';
	local num_stats: word count `stats'; 
	local layout;
	forvalues l = 1/`num_stats' {;
		local layout `"`layout' "\multicolumn{1}{c}{@}" "';
	};
	local si 3.4;
	local title "`sector_name' reaching for yield regressions";
	local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
	local prehead `"prehead(`table_preamble' `groups')"';			
	local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
	local notes `"Notes: The gross yield is the weekly gross simple yield, expressed on an annualized basis. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor."';
	local filename `"`sector'-reaching-DiD-regressions`OUTPUT_PS'"';
	local filenames `"`filenames' `filename'`c'.tex"';
	esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls', relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
	estimates drop _all;

};

EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;


/*************************************************************************************************************
10. Reaching regressions, DiD-IV and panel-IV
*************************************************************************************************************/
#delimit;
estimates drop _all;
capture drop TaxFreeX* PrimeX* AnyInstitutionalX*;
capture drop HoldingRisk;
local sector MMF;
local sector_name `"Money market fund"';
local grossyield 7GDSYwkly;
local grossreturn 1MoGSARE;

local control_list `"TaxFree Prime AnyInstitutional"';
local allcontrols `"`control_list'"';
local endogenous `"IncdExpenses2006"';
local instruments `"IncdInstrument2006_2005"';
local varlabels `"IncdExpenses2006 `"2006 incurred expenses (IV with 2005 value)"'"';
forvalues yyyy = 2007/2013 {;
	local endogenous `"`endogenous' IncdExpensesX`yyyy'"';
	local varlabels `"`varlabels' IncdExpensesX`yyyy' `"`yyyy' incurred expenses (IV with 2005 value)"'"';
	local instruments `"`instruments' IncdInstrument`yyyy'_2005"';
	foreach control of local control_list {;
		qui gen `control'X`yyyy' = `control' * (year==`yyyy');
		local allcontrols `"`allcontrols' `control'X`yyyy'"';
	};
};

qui gen HoldingRisk = ForeignBankObligations - Repos;
qui xi i.daten, prefix(_W);
qui xi i.year, prefix(_Y);


foreach spec in `"FE"' {;* `"DiD"';

	if `"`spec'"'==`"DiD"' {;
		local controls_note `"Regressions also include categorical variables, interacted with year, for fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for whether the fund has any institutional shares."'; 
	};
	else if `"`spec'"'==`"FE"' {;
		local allcontrols `""';
		local endogenous = regexr(`"`endogenous'"',`"IncdExpenses2006"',`"IncdExpensesX2006"');
		local controls_note `"Regressions also include fund fixed effects."';
	};

	local c; local filenames;
	foreach subset in "" Prime {;
		local nregs; local weeks_indicate; local years_indicate;
		local c = `c' + 1;
		
		foreach measure in Folio_`grossyield' Folio_`grossreturn'_sd HoldingRisk WAMwk {;

			preserve;

			*qui keep if year==2006 | inrange(year,2010,2013);
			
			local controls `"`allcontrols'"';
			
			local condition; local sample_note;
			if `"`subset'"'==`"Prime"' | `"`measure'"'==`"HoldingRisk"' {;
				local condition `" if Prime"';
				foreach control in TaxFree Prime {;	
					local controls = regexr(`"`controls'"',`"`control'"',`""');
					local controls = regexr(`"`controls'"',`"(`control'X20[01][0123456789] )+"',`""');
				};
				if `"`subset'"'==`"Prime"' {;
					local sample_note `" Sample includes only Prime funds."';
					local controls_note = subinstr(`"`controls_note'"',`"fund category (tax free, prime, or U.S. government and agency securities and backed repo) and for"',`""',1);
				};
			};
			if `"`measure'"'==`"Folio_`grossreturn'_sd"' {;
				if `"`subset'"'==`"Prime"' {;
					local condition `"`condition' & tag"';
				};
				else {;
					local condition `"if tag"';
				};
				local controls = regexr(`"`controls'"',`"_Idaten\*"',`"_Iyear*"');
				local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{No}"';
				local years_indicate `"`years_indicate' & \multicolumn{1}{c}{Yes}"';
				local controls `"`controls' _Yyear*"';
			};
			else {;
				local weeks_indicate `"`weeks_indicate' & \multicolumn{1}{c}{Yes}"';
				local years_indicate `"`years_indicate' & \multicolumn{1}{c}{No}"';
				local controls `"`controls' _Wdaten*"';
			};
			
			if `"`spec'"'==`"DiD"' {;
				qui eststo: ivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', cluster(id_complex) partial(`controls') first;
			};
			else if `"`spec'"'==`"FE"' {;
				qui eststo: xtivreg2 `measure' `controls' (`endogenous' = `instruments') `condition', fe cluster(id_complex) partial(`controls') first;			
			};
			
			matrix first = e(first);
			matrix F = first["F",4];
			qui estadd scalar Fstat = F[1,1];
			qui estadd scalar KP = e(widstat);	
			local nregs = `nregs' + 1;
			
			restore;
		};

		local weeks_indicate `""Week FE `weeks_indicate'\\""';	
		local years_indicate `""Year FE `years_indicate'\\""';			
		local indicate `" `weeks_indicate' `years_indicate' "';					
		local groups1 `"& \multicolumn{`nregs'}{c}{Dependent variable:}"'; 
		local midrules1 `"\cmidrule(lr{.75em}){2-`=`nregs'+1'}"';
		local groups2 `"& \multicolumn{1}{c}{Gross} & \multicolumn{1}{c}{Std. dev.} & \multicolumn{1}{c}{Risky asset} & \multicolumn{1}{c}{Average}"'; 
		local groups3 `"& \multicolumn{1}{c}{yield} & \multicolumn{1}{c}{return} & \multicolumn{1}{c}{allocation} & \multicolumn{1}{c}{maturity}"'; 
		local groups `" "`groups1'\\ `midrules1'" "`groups2'\\" "`groups3'\\" "';	
		local stats "N_clust N";
		local stats_fmt "%12.0fc %12.0fc";
		local stats_label `" `"Fund sponsor clusters"' `"Observations"' "';
		local num_stats: word count `stats'; 
		local layout;
		forvalues l = 1/`num_stats' {;
			local layout `"`layout' "\multicolumn{1}{c}{@}" "';
		};
		local si 3.4;
		local title "`sector_name' reaching for yield regressions";
		local table_preamble `" "\begin{table}[htbp]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`nregs'}{S}}" "\hline\hline" "';
		local prehead `"prehead(`table_preamble' `groups')"';			
		local posthead `"posthead(`"\hline"' `"\multicolumn{`=`nregs'+1'}{l}{Right hand side variables:}\\"' `"\\"')"';
		local notes `"Notes: The gross yield is the weekly gross simple yield, expressed on an annualized basis. The standard deviation is the annual standard deviation of the monthly excess return, defined as the gross return less the 1 month TBill. The risky asset allocation equals the asset share in foreign bank obligations less the share in repo, and for this regression the sample excludes non-Prime funds. The average maturity refers to the average of the maturity of a fund's securities, in days. `controls_note'`sample_note' Standard errors in parentheses and clustered by fund sponsor."';
		local filename `"`sector'-reaching-`spec'-IV-regressions`OUTPUT_PS'"';
		local filenames `"`filenames' `filename'`c'.tex"';
		esttab * using "$OutputPath/`filename'`c'.tex",  replace cells(b(star fmt(%9.3f)) se(par fmt(%9.3f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01) drop(_cons `controls' *#daten, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
		estimates drop _all;

	};
	EmbedTex `"`filenames'"' using `"$OutputPath/`sector'-reaching-regressions.tex"', title(`sector' reaching regressions) author(Gabriel Chodorow-Reich) append;
};



