/*************************************************************************************************************
This ado-file constructs QE announcement window changes of CDS prices.


Options:
sectype: Maturity of Treasury
lineplots: Make figures with intraday line plot by institution
*************************************************************************************************************/
capture program drop EventStudyCDS
program define EventStudyCDS
version 12.1
syntax name(name=sector id="Sector LI or BHC"), [lower(integer 2)] [upper(integer 18)] [sectype(integer 5)] [tenor(integer 5)] [lineplots] [paperplot] [alldates]


clear
set more off
#delimit;
tempfile data;


local LI_name `"Insurance company"';
local BHC_name `"Bank"';
local LI_cluster;
local LI_cluster `"cluster(date)"';
local BHC_cluster `"cluster(date)"';
local measure `"CDS"';
local idvar ticker;

/*************************************************************************************************************
1. Load data
*************************************************************************************************************/
qui unzipfile "$DataPath/markit_`sector'", replace;
use markit_`sector', clear;
erase markit_`sector'.dta;
keep if tier==`"SNRFOR"';
assert contributor=="Composite";
qui egen numnotmissing = rownonmiss(spread*);
drop if numnotmissing==0;
duplicates report date ticker docclause ccy;
/*After trimming observations with all spreads missing, date-ticker-docclause-ccy uniquely define observations*/
drop if missing(spread`tenor');
qui egen id = group(ticker docclause ccy);
qui tsset id date, daily;
foreach spread of varlist spread* {;
	qui replace `spread' = `spread' * 10^4; /*Convert to units of basis points*/
};
qui gen double D2spread`tenor' = F.spread`tenor' - L.spread`tenor'; /*2 day change*/
qui replace D2spread`tenor' = F.spread`tenor' - L3.spread`tenor' if missing(D2spread`tenor') & dow(date)==1; /*If event date on Monday, use prior Friday for prior observation*/
qui replace D2spread`tenor' = F3.spread`tenor' - L.spread`tenor' if missing(D2spread`tenor') & dow(date)==5; /*If event date on Friday, use following Monday for post observation*/
qui gen double D1spread`tenor' = spread`tenor' - L.spread`tenor'; /*1 day change*/
qui replace D1spread`tenor' = spread`tenor' - L3.spread`tenor' if missing(D1spread`tenor') & dow(date)==1; /*If event date on Monday, use prior Friday for prior observation*/
qui replace ticker = `"MS"' if ticker==`"MWD"';

/*Merge Bloomberg CDS data*/
preserve;
qui use "$SavePath/CDS_`sector'-Bloomberg", clear;
qui keep date ticker source PX_LAST;
qui replace ticker = `"C"' if ticker==`"CINC"';
qui replace ticker = `"JPM"' if ticker==`"JPMCC"';
qui replace ticker = `"COF"' if ticker==`"CCOF"';
qui replace ticker = `"WFC"' if ticker==`"WELLFARGO"';
rename PX_LAST spread5;
qui reshape wide spread5, i(date ticker) j(source) string;
qui gen ccy="USD";
tempfile Bloomberg;
qui save `Bloomberg';
restore;
merge m:1 date ticker using `Bloomberg', nogenerate;
qui gen DintradayspreadNL = spread5CBGN - spread5CBGL;
qui gen DintradayspreadNT = spread5CBGN - spread5CBGT;
qui gen DintradayspreadLT = spread5CBGL - spread5CBGT;
qui drop if inlist(ticker,`"AIG"'); /*AIG is weird for obvious reasons.*/

qui gen QE = "";
qui gen time = .;
qui format time %tCHH:MM;
do QE_dates;
forvalues q = 1/4 {;
	local i = 1;
	while `"${QE`q'_`i'}"'!=`""' {; 
		if `"`alldates'"'==`""' & inlist(`"${QE`q'_`i'}"',`"25nov2008 8:15:00"',`"12aug2009 14:16:10"',`"4nov2009 14:19:00"',`"27aug2010 10:00:00"',`"3nov2010 14:16:39"',`"31aug2012 10:00:00"',`"12dec2012 12:30:02"',`"18dec2013 14:00:00"') {; 
			local i = `i'+1;
			continue; /*Dates without any "action" in the Treasury market*/
		};
		else {;
			qui replace QE = `"QE`q'_`i'"' if date==dofC(`=tC(${QE`q'_`i'})');
			local time: word 2 of ${QE`q'_`i'};
			qui replace time = tC(`time') if QE == `"QE`q'_`i'"';
			local datelist `"`datelist' QE`q'_`i'"';
			local i = `i'+1;
		};
	};
};
qui gen eventdate = .;
foreach date of local datelist {;
	qui replace eventdate = 1 if date==dofC(Clock(`"${`date'}"',`"DMYhms"'));
};

/*Merge change in yield to maturity of on the run 5 year Treasury. Use intraday window for Treasury because noise in RHS measure would introduce attenuation bias*/
preserve;
qui use `"$DataPath/Treasuries-event-study-on-the-run"' if sectype==`sectype' & !missing(datetime), replace;
eventtime datetime, lower(`lower') upper(`upper') band(0:04:59) gen(eventtime);
qui replace eventtime =  1 if tin(10jul2013 17:25,10jul2013 17:29:00) & `upper'>40;
qui drop if missing(eventtime);
qui collapse (mean)  ytm, by(date eventtime);
qui tsset date eventtime, delta(1);
foreach var of varlist ytm {;
	qui gen D`var' = S2.`var';
};
qui keep if eventtime == 1;
qui gen DytmBP = 10^4 * Dytm; /*Converts units to basis points*/
qui gen DytmBP10 = 10^3 * Dytm; /*Converts units to 10 basis points*/
label variable DytmBP10 `"10 basis point change in `sectype' year Treasury"';
qui gen double DytmBP10_pos = (Dytm>0) if !missing(Dytm);
label variable DytmBP10_pos `"Increase in `sectype' year Treasury"';
qui gen double DytmBP10_neg = (Dytm<0) if !missing(Dytm);
label variable DytmBP10_neg `"Decrease in `sectype' year Treasury"';
qui save `data';
restore;
merge m:1 date using `data', nogenerate keep(master matched) keepusing(*ytm*);

/*Quasi-intraday changes. CBGN, CBGL, CBGT report data as of 17:15 in New York, London, and Tokyo, respectively. For the dates I use,
I don't need to worry about differences in daylight savings time adjustment dates*/
qui gen Dintradayspread = DintradayspreadNL if time>tC(12:15:00) & !missing(time);
qui replace Dintradayspread = DintradayspreadLT if time<tC(12:15:00);


/*Merge data from CRSP daily stock file*/
preserve;
dsf `sector'; /*See program at end of file*/
foreach pair in "BOFA BAC" {;
	tokenize `pair';
	qui replace ticker = "`1'" if ticker=="`2'";
};
tempfile dsf;
qui save `dsf', replace;
restore;
merge m:1 date ticker using `dsf', nogenerate keep(master matched);

/*************************************************************************************************************
2. Diagnostics and sample restriction
*************************************************************************************************************/
preserve;
qui keep if eventdate==1;
foreach var of varlist ticker date docclause {;
	qui levelsof `var', local(`var's);
};
foreach docclause of local docclauses {;
	di `"`docclause'"';
	table date ticker if ccy=="USD" & docclause=="`docclause'", c(count D1spread);
};
restore;
/*I lose no date-ticker observations by restricting to USD currency and MR restructuring clause*/
qui keep if ccy=="USD" & (docclause=="MR" | missing(docclause));
qui drop id;
qui encode ticker, gen(id);
qui tsset id date;

/*************************************************************************************************************
3. Regressions
*************************************************************************************************************/
preserve;
qui keep if eventdate==1;
local n = 0;
foreach depvar in D1spread`tenor' Dintradayspread {;
	foreach rhs in `"c.DytmBP10#ibn.id"' `"DytmBP10"' `"DytmBP10_pos DytmBP10_neg"' `"DytmBP10 if substr(QE,3,1)!="1""' `"DytmBP10_pos DytmBP10_neg if substr(QE,3,1)!="1""' {;
		if `"`rhs'"'==`"c.DytmBP10#ibn.id"' & `"`sector'"'=="BHC" {;
			continue; /*Too many BHCs to show estimates separately by BHC*/
		};
		
		local cons;
		if regexm(`"`rhs'"',`"^DytmBP10_pos DytmBP10_neg"') {;
			local cons `"noconstant"';
		};
		
		qui eststo: reg `depvar' `rhs', `cons' robust ``sector'_cluster';
		
		local n = `n' + 1;	
		if `"`rhs'"'==`"c.DytmBP10#ibn.id"' {;
			local bnames: colnames e(b);
			foreach bname of local bnames {; /*Change factor variables to have labels in output*/
				if regexm(`"`bname'"',`"^[1-9][0-9]?[b]?[n]?\.id"') & `"`depvar'"'==`"D1spread`tenor'"' {; 
					if regexm(`"`bname'"',`"^[1-9][0-9]?bn\.id"') {; /*Stata is confused on whether bn belongs in the name. It is in colnames e(b), but not in the stored output.*/
						local bname = regexr(`"`bname'"',`"bn.id"',`".id"');
					};
					local id = regexr(`"`bname'"',`"\.id.*"',`""');
					local idlabel: label id `id';
					local cvarlabel `"10 b.p. change"';
				};
				local varlabels `"`varlabels' `bname' `"`idlabel' X `cvarlabel'"'"';
			};
		};
		if strmatch(`"`rhs'"',`"* if *"') {;
			local indicate `"`indicate' & \multicolumn{1}{c}{No}"';
		};
		else {;
			local indicate `"`indicate' & \multicolumn{1}{c}{Yes}"';
		};		
	};
};

local indicate `"`"Includes QE1 `indicate' \\"'"';
local groups1 `"& \multicolumn{`n'}{c}{Dependent variable: change in CDS spread, basis points}"'; 
local midrules1 `"\cmidrule(l{.75em}){2-`=`n'+1'}"';
local groups2 `"& \multicolumn{`=`n'/2'}{c}{1 day change} & \multicolumn{`=`n'/2'}{c}{Intraday change}"'; 
local midrules2 `"\cmidrule(l{.75em}){2-`=`n'/2+1'} \cmidrule(l{.75em}){`=`n'/2+2'-`=`n'+1'}"';
local groups `"" `groups1'\\ `midrules1' `groups2'\\ `midrules2' ""';	
if e(N_clust)!=. {;
	local stats "r2 N_clust N";
	local stats_fmt "%9.3f %9.0f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Clusters"' `"Observations"' "';
};
else {;
	local stats "r2 N";
	local stats_fmt "%9.3f %9.0f";
	local stats_label `" `"LatexMath R^2$"' `"Observations"' "';
};
local num_stats: word count `stats'; 
local layout;
forvalues l = 1/`num_stats' {;
	local layout `"`layout' "\multicolumn{1}{c}{@}" "';
};
local si 3.3;
local title "``sector'_name' CDS response";
local table_preamble `" "\begin{table}[htb]\centering \def\sym#1{\ifmmode^{#1}\else\(^{#1}\)\fi}" "\caption{`title'}" "\begin{tabularx}{\hsize}{@{\hskip\tabcolsep\extracolsep\fill}l*{`n'}{S}}" "\hline\hline" "';
local prehead `"prehead(`table_preamble' `groups')"';			
local posthead `"posthead(`"\hline"' `"\multicolumn{`=`n'+1'}{l}{Right hand side variable:}\\"' `"\\"')"';
local notes `"Notes: The dependent variable is the change in the 5 year tenor CDS. The variable change in `sectype' year Treasury is the change in the yield to maturity of the on-the-run `sectype' year Treasury from `lower' minutes before to `upper' minutes after the announcement. Heteroskedastic robust standard errors in parentheses."';
local filename `"`sector'-`measure'-response-window-`lower'-`upper'`OUTPUT_PS'"';
local filenames `"`filenames' `filename'"';
esttab * using "$OutputPath/`filename'.tex", replace cells(b(star fmt(%9.1f)) se(par fmt(%9.1f) abs)) starlevels(\$^{+}$ 0.1 \$^{*}$ 0.05 \$^{**}$ 0.01)  drop(_cons, relax) `prehead' `posthead' order() label varlabel(`varlabels') postfoot(`"\hline\hline \multicolumn{@span}{l}{\begin{minipage}{\hsize} \rule{0pt}{9pt} \footnotesize `notes'  \end{minipage} }\\ \end{tabularx} \end{table}"') stats(`stats', layout(`layout') fmt(`stats_fmt') labels(`stats_label')) collabels(,none) numbers nomtitles substitute(LatexMath \$ _ "-" sub _ # " X " tabular* tabularx "{c}}" "{S}}" "{X}}" "{S}}" "\centering" "\centering \sisetup{table-format=`si'}") width(\hsize) prefoot(`indicate' "\label{tab:`filename'}");	
EmbedTex `"`filename'"' using `"$OutputPath/`filename'-embedded.tex"', title(CDS price response) author(Gabriel Chodorow-Reich) landscape replace;
estimates drop _all;

/*Scatterplot with QE rounds labeled*/
local depvar Dintradayspread;
local devar_desc `"Change in CDS, basis points"';
local labsize medsmall;
local colorj = 0; local symbolj = 0; local j = 0;
local colors `"blue red green brown"';
local symbols `"O D T S"';
forvalues q = 1/4 {;
	local j = `j' + 1;
	foreach item in color symbol {;
		local `item'j = ``item'j' + 1;
		local `item': word ``item'j' of ``item's';
		if `"`item'"'==`""' {;
			local `item'j = 1;
			local `item': word ``item'j' of ``item's';
		};
	};
	local scatters `"`scatters' (scatter `depvar' DytmBP if substr(QE,3,1)==`"`q'"', msymbol(`symbol') msize(vsmall) mcolor(`color'))"';
	local legendkey `"`legendkey' label(`j' `"QE `j'"')"';
};
local legend `"on row(`=round(`j'/2)') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
twoway `scatters', subtitle("`depvar_desc'", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize')) xlabel(,tposition(outside) labsize(`labsize')) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot-window-`lower'-`upper'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures `lower' to `upper' minute window.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) replace;

/*Mean of market cap*/
qui egen marketcapbar = mean(marketcap), by(`idvar');

/*Add combined dates for median, mean, value-weighted mean*/
qui levelsof `idvar', local(`idvar's);
foreach pair in `"16dec2008 18mar2009"' `"22may2013 19jun2013"' `"10jul2013 18sep2013"' {;
	local d = `d' + 1;
	tokenize `pair';
	foreach `idvar' of local `idvar's {;
		qui set obs `=`=_N'+1';
		qui replace date = `d' in `=_N';
		foreach var of varlist `depvar' D1spread5 DytmBP {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(sum) if r(N)!=0 in `=_N';
		};
		qui replace DytmBP = . in `=_N' if r(N)!=2;
		foreach var of varlist marketcap* {;
			qui sum `var' if inlist(date,td(`1'),td(`2')) & `idvar' == `"``idvar''"';
			qui replace `var' = r(mean) in `=_N';
		};
		qui replace `idvar' = `"``idvar''"' in `=_N';
	};
};

/*Add median, mean, value-weighted mean, and p-values of 2-sided t-test on each date to data set*/
qui levelsof date, local(dates);
qui gen Pvalue`depvar' = .;
qui gen PvalueD1spread5 = .;
foreach date of local dates {;
	foreach stat in Median Mean vwMean {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace ticker = `"`stat'"' in `=_N';
		sum DytmBP if date == `date', meanonly;
		qui replace DytmBP = r(mean) in `=_N';
		foreach var of varlist `depvar' D1spread5 {;
			if `"`stat'"'==`"Median"' {; 
				qui sum `var' if date == `date' & !inlist(ticker,`"Median"',`"Mean"',`"vwMean"'), detail;
				qui replace `var' = r(p50) in `=_N';
			};
			else {;
				if `"`stat'"'==`"Mean"' {;
					local wgt `""';
				};
				else if `"`stat'"'==`"vwMean"' {;
					local wgt `"[aw=marketcapbar]"';
				};
				foreach vce in "" ", robust" {;
					capture qui reg `var' if date==`date' & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"') `wgt' `vce';
					if _rc==0 {;
						qui replace `var' = _b[_cons] in `=_N';
						qui test _cons;
						/*Use maximum of conventional and robust SE*/
						qui replace Pvalue`var' = r(p) in `=_N' if (r(p)>Pvalue`var'[`=_N'] & !missing(r(p))) | missing(Pvalue`var'[`=_N']);
					};
				};
			};
		};
	};
};
rename Pvalue`depvar' Pvalue;

/*Drop combined dates with individual tickers*/
qui drop if inrange(date,1,3) & !inlist(`idvar',`"Median"',`"Mean"',`"vwMean"');

/*Drop combined dates with summary measures and add back later*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		foreach var of varlist `depvar' D1spread5 DytmBP Pvalue* {;
			sum `var' if `idvar'==`"`stat'"' & date==`date', meanonly;
			local `stat'`var'`date' = r(mean);
		};
	};
};
qui drop if inrange(date,1,3);


/*Drop outliers for readability of graph*/
if `"`depvar'"'==`"D1spread`tenor'"' {;
	if `"`sector'"'==`"LI"' {;
		foreach pair in `"LNC 18mar2009"' {;
			tokenize `pair';
			sum `depvar' if ticker==`"`1'"' & date==td(`2'), meanonly;
			local datestring = string(td(`2'),`"%tdMonth"') + " " + string(td(`2'),`"%tddd"') + ", " + string(td(`2'),`"%tdCCYY"');
			local dropdates `"`dropdates', `1' on `datestring' (+`=round(r(mean))')"';
			qui drop if ticker==`"`1'"' & date==td(`2');
		};
		local dropdates = regexr(`"`dropdates'"',`"^, "',`""');
		local dropdates `" The figure omits the CDS change of `dropdates' to enhance readability."';
	};
};

/*Scatterplot with dates labeled*/
local depvar_desc `"Change in CDS, basis points"';
local labsize medsmall;
local altangle1 75; /*Put some date labels at an alternate label to enhance readability*/
local altangle2 105; /*Put some date labels at an alternate label to enhance readability*/
qui levelsof DytmBP, local(values);
foreach value of local values {;
	sum date if round(DytmBP,0.0001)==round(`value',0.0001), meanonly;
	if inlist(r(mean),`=td(9aug2011)',`=td(1dec2008)',`=td(13sep2012)') {;
		local tlabels`altangle1' `"`tlabels`altangle1'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else if inlist(r(mean),`=td(10aug2010)',`=td(18sep2013)',`=td(23sep2009)',`=td(22may2013)') {;
		local tlabels`altangle2' `"`tlabels`altangle2'' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
	else {;
		local tlabels `"`tlabels' `value' `"`=string(r(mean),`"%tdnn/dd/YY"')'"'"';
	};
};
local xlabsize small; 
local xlabels `"xlabel(`tlabels', tposition(outside) labsize(`xlabsize') angle(vertical))"';
foreach a in `altangle1' `altangle2' {;
	local xlabels `"`xlabels' xlabel(`tlabels`a'', tposition(outside) labsize(`xlabsize') angle(`a') custom add)"';
};
if inlist(`"`sector'"',`"LI"',`"BHC"') {;
	if `"`sector'"'==`"LI"' {;
		local legend_p = 5;
	};
	else if `"`sector'"'==`"BHC"' {;
		local legend_p = 11;
	};
	local colorj = 0; local symbolj = 0; local j = 0; local scatters; local legendkey;
	local colors `"blue red green black brown purple brown cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
	local symbols `"D T S + X"';
	qui levelsof ticker if !inlist(ticker,`"Median"',`"Mean"',`"vwMean"') &!missing(`depvar'), local(tickers);
	foreach ticker of local tickers {;
		local j = `j' + 1;
		foreach item in color symbol {;
			local `item'j = ``item'j' + 1;
			local `item': word ``item'j' of ``item's';
			if `"`item'"'==`""' {;
				local `item'j = 1;
				local `item': word ``item'j' of ``item's';
			};
		};
		local scatters `"`scatters' (scatter `depvar' DytmBP if ticker==`"`ticker'"', msymbol(`symbol') msize(small) mcolor(`color') xaxis(1 2))"';
		local legendkey `"`legendkey' label(`j' `"`ticker'"')"';
	};
	local j = `j'+1;
	local legendkey `"`legendkey' label(`j' `"Median"')"';
	local legend `"on row(`=round(`j'/2)') rowgap(0) size(`labsize') position(`legend_p') region(lcolor(white)) ring(0) `legendkey'"';
};
else {;
	local scatters `"(scatter `depvar' DytmBP, msymbol(O) msize(small) mcolor(blue) xaxis(1 2))"';
	local legend `"off"';
	local scatters `"`scatters' (scatter `depvar' DytmBP if ticker==`"Median"', msymbol(O) msize(medsmall) mcolor(black) xaxis(1 2))"';
};
twoway `scatters', yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize(3.5) xsize(6);
local filename `sector'-`measure'-event-study-scatterplot2-window-`lower'-`upper'`alldates'; 
qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
qui graph export `"$PaperPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response to unconventional monetary policy surprises"';
local label `"`filename'"';
local fullnote `"Notes: The change in CDS is the change in the `tenor' year spread, Tokyo close to London close (announcement before 12:15pm) or London close to New York close (announcement after 12:15pm), on the announcement date. The change in the on-the-run `sectype' year Treasury is the change in the yield to maturity from `lower' minutes before to `upper' minutes after the announcement.`dropdates'"';
TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!ht) replace;
TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(fig:`label') note(`fullnote') position(!p) replace;
EmbedTex `"`sector'_figures/`filename'"' using `"$OutputPath/`sector' `measure' price response figures `lower' to `upper' minute window.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) append;

/*Presentation figure*/
local labsize medlarge;
local xlabsize medium; 
local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',1);
local xlabels = subinstr(`"`xlabels'"',`"small"',`"`xlabsize'"',.);
twoway `scatters', yline(0, lpattern(dash) lwidth(thin)) xline(0, lpattern(dash) lwidth(thin)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(outline) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("`depvar_desc'", size(`labsize')) xtitle("Announcement date", size(`labsize')) xtitle("Change in on-the-run `sectype' year Treasury, basis points", size(`labsize') axis(2)) `xlabels' xlabel(,tposition(outside) labsize(`labsize') axis(2)) ysize($ysize_p) xsize($xsize_p);
qui graph export `"$PresentationPath/`filename'.eps"', replace;
local title `"``sector'_name' `measure' price response"';
TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;


/*Add back combined dates with summary measures*/
forvalues date = 1/3 {;
	foreach stat in `"Median"' `"Mean"' `"vwMean"' {;
		qui set obs `=`=_N'+1';
		qui replace date = `date' in `=_N';
		qui replace `idvar' = `"`stat'"' in `=_N';
		foreach var of varlist `depvar' D1spread5 DytmBP Pvalue* {;
			qui replace `var' = ``stat'`var'`date'' in `=_N';
		};
	};
};

qui save "$DataPath/`sector'-`measure'-eventtime-window-`lower'-`upper'`alldates'", replace;

restore;

/*************************************************************************************************************
4. Figures of CDS over medium horizon
*************************************************************************************************************/
if `"`paperplot'"'==`"paperplot"' {;

	preserve;
	qui tsfill;
	qui decode id, gen(ticker2);
	qui replace ticker = ticker2 if missing(ticker);
	qui drop ticker2;

	local depvar spread`tenor'y;
	local depvar_desc `"`tenor' year tenor, basis points"';
	local labsize medsmall;
	local tfmt `"%tdMon-YY"';
	local legend off;
	local filenames;
	local tlabels;
	forvalues yyyy = 2008/2013 {;
		local tlabels `"`tlabels' `=td(1jan`yyyy')'"';
	};

	if inlist(`"`sector'"',`"LI"',`"BHC"') {;
		local colorj = 0; local symbolj = 0; local j = 0; local lines; local legendkey;
		local colors `"blue red green black brown purple brown cranberry cyan dkgreen dknavy dkorange gold ltblue magenta maroon olive pink sand teal yellow"';
		local patterns `"dot longdash shortdash solid longdash_dot shortdash_dot"';
		if `"`sector'"'==`"BHC"' {;
			qui keep if inlist(ticker,`"BOFA"',`"C"',`"GS"',`"JPM"',`"MS"',`"USB"',`"WFC"'); /*Dropping ALLY from the graph because of the GM implications*/
			qui replace `depvar' = spread`tenor'CBGN if inlist(ticker,`"BOFA"'); /*Corrects a mistake I made in not noticing incompatibility of tickers in markit and crsp*/
			local colors = regexr(`"`colors'"',`"blue"',`""');
			local patterns = regexr(`"`patterns'"',`"dot"',`""');
		};
		qui levelsof ticker if !missing(`depvar'), local(tickers);
		foreach ticker of local tickers {;
			local j = `j' + 1;
			foreach item in color pattern {;
				local `item'j = ``item'j' + 1;
				local `item': word ``item'j' of ``item's';
				if `"`item'"'==`""' {;
					local `item'j = 1;
					local `item': word ``item'j' of ``item's';
				};
			};
			local lines `"`lines' (tsline `depvar' if ticker==`"`ticker'"', lpattern(`pattern') lwidth(medium) lcolor(`color') cmissing(no))"';
			local legendkey `"`legendkey' label(`j' `"`ticker'"')"';
		};
		local j = `j'+1;
		local legend `"on row(`=round(`j'/2)') rowgap(0) size(`labsize') position(1) region(lcolor(white)) ring(0) `legendkey'"';
		local fullnote `""';
	};
	else {;
		/*Add median, p50, p90 on each date to data set*/
		local stats `"p10 p50 p90"';
		qui levelsof date, local(dates);
		foreach date of local dates {;
			qui sum `depvar' if date==`date', detail;
			foreach stat of local stats {;
				qui set obs `=`=_N'+1';
				qui replace date = `date' in `=_N';
				qui replace ticker = `"`stat'"' in `=_N';
				qui replace `depvar' = r(`stat') in `=_N';
			};
		};
		foreach stat of local stats {;
			if inlist(`"`stat'"',`"p50"') {;
				local pattern solid;
				local color blue;
			};
			else {;
				local pattern dot;
				local color red;
			};
			local lines `"`lines' (tsline `depvar' if ticker==`"`stat'"', lpattern(`pattern') lwidth(medium) lcolor(`color') cmissing(no))"';
		};
		local legend `"off"';
		local fullnote `"Notes: The figure shows the median CDS spread (blue solid line) along with the 10th and 90th percentile CDS spreads (red dashed lines) for all bank holding companies with valid quotes on each date."'; 
	};
	twoway `lines' if year(date)>=2008, subtitle(`"`depvar_desc'"', size(`labsize') position(11) span)  legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("", size(`labsize')) ttitle("", size(`labsize')) tlabel(`tlabels', tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
	local filename `sector'-`measure'-time-series; 
	qui graph export `"$PaperPath/`filename'.eps"', replace;
	local title `"``sector'_name' `measure'"';
	local label `"`filename'"';
	TexFigure `"input-files/`filename'.eps"' using `"$PaperPath/`filename'.tex"', headtitle(`title') label(fig:`label') note(`fullnote') position(!ht) replace;

	/*Presentation figure*/
	local labsize medlarge;
	local legend = subinstr(`"`legend'"',`"medsmall"',`"`labsize'"',1);
	twoway `lines' if year(date)>=2008, subtitle(`"`depvar_desc'"', size(`labsize') position(11) span)  legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("", size(`labsize')) ttitle("", size(`labsize')) tlabel(`tlabels', tposition(outside) labsize(`labsize') format(`tfmt')) ysize($ysize_p) xsize($xsize_p);
	qui graph export `"$PresentationPath/`filename'.eps"', replace;
	TexFigure `"input-files/`filename'.eps"' using `"$PresentationPath/`filename'.tex"', headtitle(`title') label(fig:`label') note() position(pt) replace;

	restore;
};

if `"`lineplots'"'==`"lineplots"' {;

	local labsize medsmall;
	local tfmt `"%tdMon-YY"';
	local legend off;
	local filenames;
	local tlabels;
	qui levelsof ticker, local(tickers);
	forvalues yyyy = 2008/2013 {;
		local tlabels `"`tlabels' `=td(1jan`yyyy')'"';
	};
	local j = 0;
	foreach ticker of local tickers {;
		twoway (tsline spread`tenor'y, lpattern(solid) lwidth(medthick) lcolor(blue)) if ticker==`"`ticker'"', subtitle("CDS spread, basis points", position(11) span size(`labsize') color(black)) legend(`legend') scheme(s2mono) graphregion(color(white) margin(l=`l_m_p' r=`r_m_p')) plotregion(style(none) margin(zero)) ylabel(,nogrid tposition(outside) angle(horizontal) labsize(`labsize')) ytitle("") ttitle("", size(`labsize')) tlabel(`tlabels',tposition(outside) labsize(`labsize') format(`tfmt')) ysize(3.5) xsize(6);
		local filename = `"`sector'-CDS-`ticker'"';
		qui graph export `"$OutputPath/`sector'_figures/`filename'.pdf"', replace;
		local title = `"`ticker' CDS, `tenor' year tenor"';
		local label `"`filename'"';
		TexFigure `"`sector'_figures/`filename'.pdf"' using `"$OutputPath/`sector'_figures/`filename'.tex"', headtitle(`title') label(`label') note(`fullnote') position(!t) replace;
		local filenames `"`filenames' `sector'_figures/`filename'.tex"';
		local j = `j'+1;
		if `j'>12 {;
			local filenames `"`filenames' \clearpage"';
			local j = 0;
		};
	};
	EmbedTex `"`filenames'"' using `"$OutputPath/`sector' `measure' price response figures.tex"', title(`sector' `measure' price response) author(Gabriel Chodorow-Reich) append;
};

end;



/*************************************************************************************************************
5. Called programs
*************************************************************************************************************/
capture program drop dsf;
program define dsf;
syntax name(name=sector id="Sector LI, BHC, or SP500");
qui unzipfile "$DataPath/crspq_`sector'_dsf", replace;
use crspq_`sector'_dsf, clear;
erase crspq_`sector'_dsf.dta;
qui tsset permno date;
qui replace prc = -prc if prc<0; /*Negative price indicates observation is midpoint between bid and ask*/
qui gen double D1logprice = log(prc) - log(L.prc) if cfacpr==L.cfacpr; /*1 day change*/
qui replace D1logprice = log(prc) - log(L3.prc) if missing(D1logprice) & dow(date)==1 & cfacpr==L3.cfacpr; /*If event date on Monday, use prior Friday for prior observation*/
qui gen double marketcap = prc*shrout;
#delimit;
foreach pair in "2jan2008 24nov2008" {;
	tokenize `pair';
	forvalues s = 1/2 {;
		qui egen adjprc_``s'' = max(prc/cfacpr * (date==td(``s''))), by(permno);
	};
	qui gen double Dlogprice_`1'_`2' = log(adjprc_`2'/adjprc_`1');
};
tempfile dsf;
save `dsf';
qui unzipfile "$DataPath/CRSPQ_`sector'_stocknames", replace;
qui use crspq_`sector'_stocknames;
erase crspq_`sector'_stocknames.dta;
qui egen maxdate = max(nameenddt), by(permno);
qui keep if nameenddt==maxdate;
qui drop maxdate;
merge 1:m permno using `dsf', nogenerate keep(matched);
qui drop if nameenddt<date;
qui drop if inlist(comnam,"CANNON EXPRESS INC","INVERNESS MEDICAL TECHNOLOGY IN");
qui egen maxout = max(shrout), by(permco date);
qui drop if shrout!=maxout;
qui drop maxout;
end;

