/* table6.do
This file caculates the regression coefficients, standard errors, and R-squared measures reported in Table 6 of
Campbell, Evans, Justiniano, and Fisher (2012).
These tables extend Table 5 in Gurkaynak, Sack, and Swanson (2005)
The data we use can be found in the Excel spreadsheet gssTable5Recreation.xlsx. 
*/
version 12.1

log using "table6.smcl",replace

//Set up a matrix to store regression coefficients and the R-squared measures. We give this matrix row and column names to denote the dependent variable (row) and
//coefficient estimate (column).
matrix betahat=J(5,3,0)
matrix rownames betahat=2year 5year 10year aaa baa
matrix colnames betahat= target path R2 
//For reporting purposes, we associated each row label with a longer string using the associative array (hash) |rowlabels|.
mata:
	rowlabels=asarray_create()
	asarray(rowlabels,"2year","Two-Year Note")
	asarray(rowlabels,"5year","Five-Year Note")
	asarray(rowlabels,"10year","Ten-Year Note")
	asarray(rowlabels,"aaa","Aaa Corporate Bond")
	asarray(rowlabels,"baa","Baa Corporate Bond")
end

matrix standardErrors=J(5,3,0)
matrix rownames standardErrors= 2year 5year 10year aaa baa
matrix colnames standardErrors= target path R2 

/*
	Regressions for the crisis sample period. 
*/


import excel using "gssTable5Recreation.xlsx",clear sheet("Regressions Aug07_Dec11") cellrange(A2:H41) firstrow case(lower)

// 2 Year Treasury Note
sum treasury2year target path
reg treasury2year target path, robust

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"2year"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"2year"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"2year"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"2year"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"2year"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"2year"),colnumb(betahat,"R2")]=.

//5 Year Treasury Note 
sum treasury5year target path
reg treasury5year target path, robust

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"5year"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"5year"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"5year"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"5year"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"5year"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"5year"),colnumb(betahat,"R2")]=.


//10 Year Treasury Note
sum treasury10year target path
reg treasury10year target path, robust

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"10year"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"10year"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"10year"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"10year"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"10year"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"10year"),colnumb(betahat,"R2")]=.

//Seasoned Corporate Aaa Bonds 
sum aaabonds target path
reg aaabonds target path, robust

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"aaa"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"aaa"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"aaa"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"aaa"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"aaa"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"aaa"),colnumb(betahat,"R2")]=.

//Seasoned Corporate Baa Bonds
sum baabonds target path
reg baabonds target path, robust

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"baa"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"baa"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"baa"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"baa"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"baa"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"baa"),colnumb(betahat,"R2")]=.

/*
\subsection{Calcualte significance stars.}
To calculate the estimates' t-statistics, we transfer |betahat| and |standardErrors| into mata, use element-by-element matrix division/
*/

mata:

	betahat=st_matrix("betahat")
	standardErrors=st_matrix("standardErrors")
	tStatistics=betahat:/standardErrors
	
	onePercent=	(abs(tStatistics):>2.5758) :* (standardErrors:<.)
	fivePercent=(abs(tStatistics):>1.9600) :* (standardErrors:<.)
	tenPercent=	(abs(tStatistics):>1.6449) :* (standardErrors:<.)
				
end

/*
\subsection{Create formatted output}
*/
//The code consists of an outer loop that cycles through the table's rows and an inner loop that cycles through its columns.
mata:

unlink("table6.tex")
f1=fopen("table6.tex","w")
//Get the row stripe for |betahat|, which we use to retrieve labels for the rows of estimates from |rowlabels|.
rowids=st_matrixrowstripe("betahat")
for(i=1;i<=rows(betahat);i++) {

	//First, place the row label into the table.
	thisString=asarray(rowlabels,rowids[i,2])
	fwrite(f1,thisString)
	fwrite(f1,"&")
	
	//Create the row of estimates.
	for(j=1;j<=cols(betahat);j++) {
		
		//If the coefficient is negative, put a zero-width negative sign into the output file.
		if (betahat[i,j]<0){
			fwrite(f1,"\makebox[0pt][r]{-}")
		}
		//Next, put coefficient's appropriately-rounded decimal expansion into the table (in absolute value).
		thisString=sprintf("%4.3f",abs(betahat[i,j]))
		fwrite(f1,thisString)
		
		//Place significance stars after the estimated coefficients.
		if (onePercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast\hspace{-3pt}\ast\hspace{-3pt}\ast$}}$")
		} else if (fivePercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast\hspace{-3pt}\ast$}}$")
		} else if (tenPercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast$}}$")
		}
		
		//Finally, place either a column separator or carriage return into the output file, as appropriate.
		if (j==cols(betahat)){
			fput(f1,"\\")
		} else {
			fwrite(f1,"&")
		}
		
	}
	//Create the accompanying row of standard errors.
	fwrite(f1,"& ")
	
	for(j=1;j<=cols(betahat);j++) {
		if (standardErrors[i,j]<.) {
			thisString=sprintf("\makebox[0pt][r]{(}%4.3f\makebox[0pt][l]{)}",standardErrors[i,j])
			fwrite(f1,thisString)
		}
		
		if (j==cols(betahat)){
			fput(f1,"\\ \\") //This puts a blank row between the separate regressions' results.
		} else {
			fwrite(f1,"&")
		}	
	}	

}
fclose(f1)

end



log close

