function forcOut=forcStateSub(minSim,minProblem)
%% I. Inputs are structures minSim and minProblem 

%% I.1 minSim with the following fields 

%% I.1.a General 
% .Nx: Dimension of the shocks 
% .innovMatOrig: original matrix of innovations 
% .availDataPos: position of any shocks that are available, ie. smooth 
%                will generate an error if wish to simulate shocks 
% .flagSimul: 1 if wish to generate shocks 
% 
%% I.1.b sim for simulation 
% .simMean: mean (Nx 1)  for the shocks to be simulated 
% .simVCV: VCV (Nx Nx) for the shicks to be simulated 
% .simPos: position of the non-zero shocks to be simulated  

%% I.1.c match for the targets to match 
% .innovPos: positions of the innovations that can be used to hit the
%            target 
% .Nmatch: Number of targets to match 
% .matchPos: Position in the states of the targets to match 
% .matchTarget: Values of the target states to match 

%% I.1.d Solution Matrices .G .R .Z .C .tauVecFake 

%% I.1.e Forecasting 
% .sTMinusOne: Jumping point state
% .Nforc: Number of forecasts 

%% minSim, Structure with all settings for the problem 
%% minProblem, Structure with the settings of the minimization problem 
%% 1. Generate Errors 
tempEtaMat=minSim.innovMatOrig;
if minSim.flagSimul==1    
    if isempty(minSim.availDataPos)==false 
        error('Cannot generate errors over Smooth Innovations') 
    end     
    tempEta=minSim.mean+minSim.VCV'*randn(minSim.Nx,1);
    tempEtaMat(1,minSim.pos)=tempEta(minSim.pos);
    clear tempEta;
end 
    
%% 2. Initial Values minSim.x0  
x0=minSim.innovT(minSim.innovPos); 
f0=minStateDiscrep(x0,minSim,tempEtaMat(1,:)); 

%% 3. Minimization
outMin=minStlsqnonlin(x0(:),tempEtaMat(1,:),minProblem,minSim); 

if outMin.exitflag~=1 || ( outMin.resnorm > minProblem.TolFun )
    forcOut.flagOK=0;
    return 
end 
tempEtaMat(1,minSim.innovPos)=outMin.xstar; 

%% 4. Unadjusted Forecast 
[stf,yf]=forecastLoop(minSim.G,minSim.R,minSim.Z,minSim.C,...
    minSim.tauVecFake,minSim.sTMinusOne(:),tempEtaMat,minSim.Nforc);

%% 5. Check discrepancy between conditional forecast and output from 
%  the Non-Linear Optimization 
tol=1e-4; 
if abs( norm( stf(1,minSim.matchPos)'-minSim.matchTarget(:) ) - outMin.resnorm ) > tol 
    warning ('Conditional forecast divergence with taget greater than NLLSQ') 
    forcOut.flagOK=0;
    return
end
forcOut.flagOK=1;

%% 6. Forecast Adjusted
forcAdjusted=zb_quickforecast_sub(stf,minSim.popForecast,minSim.inStru); 

%% 7. Compile Output and return 
forcOut.stF=stf; 
forcOut.yF=yf'; 
forcOut.forcAlt=forcAdjusted; 
forcOut.min=outMin; 
forcOut.etaMat=tempEtaMat; 

if minSim.flagVerbose==0 
    return 
end 
%% 8. If flagVerbose == 1 print output from the minimization 
disp('Output from Minimization'); 
dispaj('Residual Norm =',outMin.resnorm); 
dispaj('Exitflag =',outMin.exitflag); 
dispaj('New Signals'); 
printcell([minSim.shocksFree(:) num2cprec(outMin.xstar(:))]); 
