%interestRateRuleAnalysis.m
%This file runs the analysis on the interest rate rule results.

%% Import the identified forward guidance shocks.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','AR2 NUs','B24:F73');
%The imported sample begins in 1996Q1 and ends in 2008Q2 for a total of 50
%observations.
nus = cell2mat(raw);
clearvars raw;

%Express the nus in basis points.
nus=100*nus;

%% Import the treasury yield data.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','Treasury Yields','D70:F116');
%The imported sample begins in 1995Q4 and ends in 2007Q2, for a total of 47
%observations The data records the yield on the last trading day of
%the quarter. 
treasuryYields = cell2mat(raw);
DtreasuryYields=treasuryYields(2:end,:)-treasuryYields(1:end-1,:); %Change in the yield within the quarter.
clearvars raw;

%% Import corporate yields data.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','Corporate Yields','D70:E116');
%The imported sample begins in 1995Q4 and ends in 2007Q2, for a total of 47
%observations. The data reports the Aaa and Baa seasoned
%corporate yields at close on the last trading day of the quarter.
corporateYields = cell2mat(raw);
DcorporateYields=corporateYields(2:end,:)-corporateYields(1:end-1,:); %Change in the yield within the quarter.
clearvars raw

%% Import the unemployment expectations data
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','Unemployment','C34:I80');
%The imported sample begins in 1996Q1 and ends in 2007Q3, for a total of 47
%observations. The data record unemployment expectations measured at the
%beginning of the quarter.
unemploymentBlueChip = cell2mat(raw);

unemploymentForecastRevisions=unemploymentBlueChip(2:end,2:5)-repmat(unemploymentBlueChip(2:end,end),1,4)...
    -unemploymentBlueChip(1:end-1,3:6)+repmat(unemploymentBlueChip(1:end-1,end),1,4);
clearvars raw;

%% Import the actual unemployment rate.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','BLS','B151:B196');
%The imported sample begins in 1996Q1 and ends in 2007Q2, for a total of 46
%observations.
unemployment = cell2mat(raw);
unemploymentGap=unemployment-unemploymentBlueChip(1:end-1,end); %The final column of |unemploymentBlueChip| contains the long-run forecast, which we take to equal the natural rate.
clearvars raw;

%% Import the inflation forecast data.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','CPI','B34:H80');
%Just like the unemployment forecast data, the sample begins in 1996Q1 and
%ends in 2007Q3. The inflation expectations are for the overall CPI and are
%measured at the beginning of the quarter.
inflationBlueChip = cell2mat(raw);
inflationForecastRevisions=inflationBlueChip(2:end,2:5)-inflationBlueChip(1:end-1,3:6);
clearvars raw;

%% Import the CPI rate.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','BLS','C150:C196');
%The imported sample begins in 1995Q4 and ends in 2007Q2 for a total of 47
%observations. We immediately transform these into 46 inflation
%observations.
cpi = cell2mat(raw);
inflation=log(cpi(2:end))-log(cpi(1:end-1));
inflationGap=inflation-inflationBlueChip(1:end-1,end);
clearvars raw;

%% Plot of the monetary policy shock and its forecastable component.
%The "beautified" version of this plot appears as Figure 2 in the paper.
%The beautification requires external tools not included with this
%replication file. 

alpha=sum(nus,2);
alphaForward=sum(nus(:,2:end),2);
alpha=alpha(1:end-4);
alphaForward=alphaForward(1:end-4);

time=1996:0.25:2007.25;
%The complete intercept is in black and solid.
figure(1);
plot(time,alpha,'-k','LineWidth',2);
hold on
%The intercept's anticipated component is in red and dashed.
plot(time,alphaForward,'--r','LineWidth',2);
hold off

%Rearrange the nus so that each row refers to the information revealed on a
%given date. This moves the end of the sample from 2008Q2 to 2007Q2.
nus=[nus(1:end-4,1) nus(2:end-3,2) nus(3:end-2,3) nus(4:end-1,4) nus(5:end,5)];

%The text reports the standard deviations of the nus at the start of
%Section II.C. Here, we write them into a LaTeX table.

standardDeviations=std(nus);
f1=fopen('nusStandardDeviations.tex','w');
fprintf(f1,'%3.0f',standardDeviations(1));
for i=2:size(nus,2)
    fprintf(f1,'& %3.0f',standardDeviations(i));
end

fclose(f1);

%The text presents the correlation matrix for the nus at the end of Section II.C.
%Here, we write them to a LaTeX table. 
correlations=corrcoef(nus);
f1=fopen('nusCorrelations.tex','w');

for i=1:size(nus,2);
    if correlations(i,1)>0
        fprintf(f1,'$\\nu_{t,%i}$ & %3.2f ',i-1,correlations(i,1));
    else
        fprintf(f1,'$\\nu_{t,%i}$ & \\makebox[0pt][r]{-}%3.2f ',i-1,abs(correlations(i,1)));
    end
    for j=2:i
        if correlations(i,j)>=0
            fprintf(f1,'& %3.2f ',correlations(i,j));
        else
            fprintf(f1,'& \\makebox[0pt][r]{-}%3.2f ',abs(correlations(i,j)));
        end
    end
    fprintf(f1,'\\\\ \n');
    
end
fclose(f1);

%% Factor Analysis
% To estimate the factor model, we use the Statistics Toolbox function
% |factoran|. 
%The |factoran| function normalizes the data to have zero mean and unit
%variance before analysis, and it returns parameters based on that
%normalization. To undo this, we require the diagonal matrix of the data's
%standard deviations, G.
G=diag(standardDeviations);

%Estimate the factor model.
Z = zscore(nus);
[lambdaZ,psiZ,TZ,statsZ,FZ]=factoran(Z,1);

%Translate the parameter and factor estimates into non-normalized units
%(basis points).
lambdaX=G*lambdaZ;
psiX=diag(G*diag(psiZ)*G);

%The factor model's loadings are reported in Section II.D of the
%text. Here, we write the estimated loadings and the standard deviations'
%idiosyncratic errors to a table.

f1=fopen('factorModel.tex','w');
fprintf(f1,'Factor Loading & %3.0f & %3.0f & %3.0f & %3.0f & %3.0f \\\\ \n',lambdaX);
fprintf(f1,'Idiosyncratic Error''s Standard Deviation & %3.0f & %3.0f & %3.0f & %3.0f & %3.0f \\\\ \n',sqrt(psiX));
fclose(f1);

%The paper illustrates the policy acceleration factor with an impulse
%response plot based on feeding the shocks through the estimated interest
%rate rule. This appears as Figure 3. Note that this does NOT account for any endogenous feedback of
%policy shocks to inflation or unemployment.
%
%For this plot, we begin by loading the estimated interest rate rule
%coefficients.
[~, ~, raw] = xlsread('interestRateRuleData.xlsx','AR2 Estimates','B15:B19');
interestRateRuleCoefficients = cell2mat(raw);
%The coefficients are: intercept, rho_1, rho_2, phi_u, phi_pi.
clearvars raw;

%To calculate the two paths, create the companion matrix for the interest rate rule's autoregressive component.
F=[interestRateRuleCoefficients(2) interestRateRuleCoefficients(3); 1 0];
horizon=10;
%For the simulation of the policy acceleration shock, we place the sequence
%of shocks into the first row of |impulses|. The second row just creates
%conformability with |factorImps| and |F|.
factorImps=zeros(2,horizon);
impulses=[lambdaX' zeros(1,horizon-5)];
impulses=[impulses; zeros(1,horizon)];
factorImps(:,1)=impulses(:,1);
for t=2:horizon
   factorImps(:,t)=F*factorImps(:,t-1)+impulses(:,t); 
end

%The simulation of the conventional policy shock is much more
%straightforward.
shockImps=zeros(2,horizon);
shockImps(1,1)=1;
for t=2:horizon
    shockImps(:,t)=F*shockImps(:,t-1);
end

%For the plot, we scale the response to the ordinary shock so that it has
%the same value on impact as does the response to the policy acceleration
%shock.
factorImps=factorImps(1,:);
shockImps=shockImps(1,:)*factorImps(1);
%This code produces the impulse-response plot in the paper.
figure(2);
plot(0:horizon-1,factorImps,'-',0:horizon-1,shockImps,'--','color',[0.3 0.3 0.3],'LineWidth',3)
set(gca,'FontSize',18,'color',[1 1 1])
set(gcf,'color',[1 1 1]);
box off;
xlabel('Quarters Since Initial Impulse');
ylabel('Basis Points');
h1=legend('Factor Impulse','Standard Contemporaneous Impulse');
h1=legend(h1,'Location','SouthEast');
legend(h1,'boxoff')

%Figure 4 in the paper plots the identified factor. As with the plot of the
%monetary policy disturbance, its beautification relies on code not
%provided with this replication file. However, we produce a plain vanilla
%version here.
fgFactor=lambdaX(1)*FZ;
figure(3);
plot(time,fgFactor,'-k','LineWidth',2);

%% Asset Price and Forecast Revision Regressions
% Table 8 reports estimates and standard errors from regressing asset price
% changes on the five identified monetary policy shocks. The following code
% produces these and writes them to a LaTeX table.
yields=[DtreasuryYields DcorporateYields];
yieldNames=cell(5,1);
yieldNames{1}='2-Year Note';
yieldNames{2}='5-Year Note';
yieldNames{3}='10-Year Note';
yieldNames{4}='Aaa Corporate Bond';
yieldNames{5}='Baa Corporate Bond';

f1=fopen('table8.tex','w');
for j=1:5
    %Multiply the yields by 100 so that the coefficients can be read as
    %basis points.
   stats=regstats(100*yields(:,j),nus);
   fprintf(f1,yieldNames{j});
   
   %Cycle through the coefficients, reporting them with stars.
   for k=1:6
       beta=stats.beta(k);
       se=sqrt(stats.covb(k,k));
       tstat=beta/se;
       pval=2*(1-normcdf(abs(tstat)));
       if pval<0.01
           significanceString='\\makebox[0pt][l]{$^{\\ast\\ast\\ast}$} ';
       elseif pval<0.05
           significanceString='\\makebox[0pt][l]{$^{\\ast\\ast}$} ';
       elseif pval<0.10
           significanceString='\\makebox[0pt][l]{$^\\ast$} ';
       else
           significanceString=' ';
       end
       fprintf(f1,[' & %3.2f' significanceString],beta);

   end
   %End the table's row.
   fprintf(f1,'& %3.2f \\\\ \n',stats.rsquare);
   
   %Write the standard errors in parentheses below the coefficient
   %estimates.
   for k=1:6
       se=sqrt(stats.covb(k,k));
       fprintf(f1,' & \\makebox[0pt][r]{(}%3.2f\\makebox[0pt][l]{)} ',se);
   end
   fprintf(f1,' & \\\\ \n');
       
end

fclose(f1);

%Table 9 reports analogous regression estimates with unemployment and
%inflation forecast revisions as the dependent variables. The following code
%creates this LaTeX table.

revisions=[unemploymentForecastRevisions inflationForecastRevisions];
revisionNames=cell(1,8);
revisionNames{1}='$\\hat{u}^{-1}_t-\\hat{u}^0_t$';
revisionNames{2}='$\\hat{u}^0_{t+1}-\\hat{u}^1_{t+1}$';
revisionNames{3}='$\\hat{u}^1_{t+2}-\\hat{u}^2_{t+2}$';
revisionNames{4}='$\\hat{u}^2_{t+3}-\\hat{u}^3_{t+3}$';
revisionNames{5}='$\\pi^{-1}_t-\\pi^0_t$';
revisionNames{6}='$\\pi^0_{t+1}-\\pi^1_{t+1}$';
revisionNames{7}='$\\pi^1_{t+2}-\\pi^2_{t+2}$';
revisionNames{8}='$\\pi^2_{t+3}-\\pi^3_{t+3}$';

f1=fopen('table9.tex','w');
for j=1:8
    %Multiply the yields by 100 so that the coefficients can be read as
    %basis points.
   stats=regstats(100*revisions(:,j),nus);
   fprintf(f1,revisionNames{j});
   
   %Cycle through the coefficients, reporting them with stars.
   for k=1:6
       beta=stats.beta(k);
       se=sqrt(stats.covb(k,k));
       tstat=beta/se;
       pval=2*(1-normcdf(abs(tstat)));
       if pval<0.01
           significanceString='\\makebox[0pt][l]{$^{\\star\\star\\star}$} ';
       elseif pval<0.05
           significanceString='\\makebox[0pt][l]{$^{\\star\\star}$} ';
       elseif pval<0.10
           significanceString='\\makebox[0pt][l]{$^\\star$} ';
       else
           significanceString=' ';
       end
       fprintf(f1,[' & %3.2f' significanceString],beta);
   end
   %End the table's row.
   fprintf(f1,'& %3.2f \\\\ \n',stats.rsquare);
   
   %Write the standard errors in parentheses below the coefficient
   %estimates.
   for k=1:6
       se=sqrt(stats.covb(k,k));
       fprintf(f1,' & \\makebox[0pt][r]{(}%3.2f\\makebox[0pt][l]{)} ',se);
   end
   fprintf(f1,' & \\\\ \n');
       stats.fstat
end

fclose(f1);
