/*  table7Unemployment.do
This file estimates the regressions of inflation forecast revisions on the GSS monetary policy factors. 
 */
version 12.1

/* Target and Path factors are matched up with the Unemployment and CPI forecast revisions 1 month in the future in the excel sheets
	to make it easier to regress the factors on the 1 month in the future revisions. */
	
	
log using "table7Unemployment.smcl",replace


//Set up a matrix to store regression coefficients and the R-squared measures. We give this matrix row and column names to denote the dependent variable (row) and
//coefficient estimate (column).
matrix betahat=J(4,3,0)
matrix rownames betahat=u0 u1 u2 u3
matrix colnames betahat= target path R2 
//For reporting purposes, we associated each row label with a longer string using the associative array (hash) |rowlabels|.
mata:
	rowlabels=asarray_create()
	asarray(rowlabels,"u0","\hspace{12pt}Current Quarter")
	asarray(rowlabels,"u1","\hspace{12pt}Next Quarter")
	asarray(rowlabels,"u2","\hspace{12pt}2 Quarters Hence")
	asarray(rowlabels,"u3","\hspace{12pt}3 Quarters Hence")
end

matrix standardErrors=J(4,3,0)
matrix rownames standardErrors= u0 u1 u2 u3
matrix colnames standardErrors= target path R2 

/*
Regressions for the GSS sample period (January 1990 through December 2004) extended to the end of the pre-crisis period, June 2007.
These regressions appear in the first three columns of Table 2.
*/

import excel using "gssBlueChip.xlsx",clear sheet("Regressions Jan07_Dec11") cellrange(A6:N58) firstrow case(lower)
rename nowcast unowcast
drop if target == 0 & path == 0

*********** Unemployment Nowcast **********
sum unowcast target path
reg unowcast target path, robust noconstant

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"u0"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"u0"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"u0"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"u0"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"u0"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"u0"),colnumb(betahat,"R2")]=.


********** Unemployment Lead 1 **********
sum ulead1 target path
reg ulead1 target path, robust noconstant

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"u1"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"u1"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"u1"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"u1"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"u1"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"u1"),colnumb(betahat,"R2")]=.

********** Unemployment Lead 2 **********
sum ulead2 target path
reg ulead2 target path, robust noconstant

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"u2"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"u2"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"u2"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"u2"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"u2"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"u2"),colnumb(betahat,"R2")]=.


********** Unemployment Lead 3 **********
sum ulead3 target path
reg ulead3 target path, robust noconstant

matrix bhat=e(b)
matrix Vhat=e(V)

matrix betahat[rownumb(betahat,"u3"),colnumb(betahat,"target")] ///
	=bhat[1,colnumb(bhat,"target")]	
matrix betahat[rownumb(betahat,"u3"),colnumb(betahat,"path")] ///
	=bhat[1,colnumb(bhat,"path")]
matrix betahat[rownumb(betahat,"u3"),colnumb(betahat,"R2")]=e(r2)
matrix standardErrors[rownumb(betahat,"u3"),colnumb(betahat,"target")] ///
	=sqrt(Vhat[rownumb(Vhat,"target"),colnumb(Vhat,"target")])
matrix standardErrors[rownumb(betahat,"u3"),colnumb(betahat,"path")] ///
	=sqrt(Vhat[rownumb(Vhat,"path"),colnumb(Vhat,"path")])
matrix standardErrors[rownumb(betahat,"u3"),colnumb(betahat,"R2")]=.

/*
\subsection{Calcualte significance stars.}
To calculate the estimates' t-statistics, we transfer |betahat| and |standardErrors| into mata, use element-by-element matrix division/
*/

mata:

	betahat=st_matrix("betahat")
	standardErrors=st_matrix("standardErrors")
	tStatistics=betahat:/standardErrors
	
	onePercent=	(abs(tStatistics):>2.5758) :* (standardErrors:<.)
	fivePercent=(abs(tStatistics):>1.9600) :* (standardErrors:<.)
	tenPercent=	(abs(tStatistics):>1.6449) :* (standardErrors:<.)
				
end
/*
\subsection{Create formatted output}
*/
//The code consists of an outer loop that cycles through the table's rows and an inner loop that cycles through its columns.
mata:

unlink("table7Unemployment.tex")
f1=fopen("table7Unemployment.tex","w")
//Get the row stripe for |betahat|, which we use to retrieve labels for the rows of estimates from |rowlabels|.
rowids=st_matrixrowstripe("betahat")
for(i=1;i<=rows(betahat);i++) {

	//First, place the row label into the table.
	thisString=asarray(rowlabels,rowids[i,2])
	fwrite(f1,thisString)
	fwrite(f1,"&")
	
	//Create the row of estimates.
	for(j=1;j<=cols(betahat);j++) {
		
		//If the coefficient is negative, put a zero-width negative sign into the output file.
		if (betahat[i,j]<0){
			fwrite(f1,"\makebox[0pt][r]{-}")
		}
		//Next, put coefficient's appropriately-rounded decimal expansion into the table (in absolute value).
		thisString=sprintf("%3.2f",abs(betahat[i,j]))
		fwrite(f1,thisString)
		
		//Place significance stars after the estimated coefficients.
		if (onePercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast\hspace{-3pt}\ast\hspace{-3pt}\ast$}}$")
		} else if (fivePercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast\ast$}}$")
		} else if (tenPercent[i,j]==1) {
			fwrite(f1,"$^{\makebox[0pt][l]{$\ast$}}$")
		}
		
		//Finally, place either a column separator or carriage return into the output file, as appropriate.
		if (j==cols(betahat)){
			fput(f1,"\\")
		} else {
			fwrite(f1,"&")
		}
		
	}
	//Create the accompanying row of standard errors.
	fwrite(f1,"& ")
	
	for(j=1;j<=cols(betahat);j++) {
		if (standardErrors[i,j]<.) {
			thisString=sprintf("\makebox[0pt][r]{(}%3.2f\makebox[0pt][l]{)}",standardErrors[i,j])
			fwrite(f1,thisString)
		}
		
		if (j==cols(betahat)){
			fput(f1,"\\ \\") //This puts a blank row between the separate regressions' results.
		} else {
			fwrite(f1,"&")
		}	
	}	

}
fclose(f1)

end
log close
