% % ========================================================================= 
% function GSSMain.m 
% 
% Replication code to obtain the factors as in GSS 
% 
% The code can run through 3 samples the output of which 
% will be saved in a new folder. 
%
% S1: Feb 1990-Jun 2007, using 5 contracts  
%       Output Folder Name: Factors Sample 1
%
% S2: Feb 1994-Jun 2007, using 5 contracts 
%       Output Folder Name: Factors Sample 2
%
% S3: Aug 2007-Dec 2011, using 7 contracts
%       Output Folder Name: Factors Sample 3
% 
% In each folder the code will write an xls file called 
% "Sample <x> Table, containing 3 worksheets 
%
% Moments: with basic information about the factors, such as regression R2, 
% total variance shares (trace) and cross-correlations with the
% idiosyncratic residuals 
%
% Factors: with the time series of the factors 
% ID errors, with the time series of the idiosyncratic errors 
% 
% =========================================================================

%% Part I: Preliminaries
% Here, we clear the memory and add the sibling directory |utilities| to
% the path.
clear all
addpath('..\utilities')

%% *Part I. User Inputs*

%% I.1 Folder where data is contained
loadFolder=pwd;  

%% I.2 Xls Filename
fileName='GSSdata';

%% Cycle through the three samples. 
for caseSample=1:3
switch caseSample
    case 1        
        sheetName   ='fiveContractsLessSep11';
        startDate   =datenum([1990 2 1]);
        endDate     =datenum([2007 6 31]);        
    case 2
        sheetName   ='fiveContractsLessSep11';
        startDate   =datenum([1994 2 1]);
        endDate     =datenum([2007 6 31]);        
    case 3
        sheetName   ='sevenContractsAllMeetings';
        startDate   =datenum([2007 8 1]);
        endDate     =datenum([2011 12 31]);
end

%% *Part II. Data and sample extraction*
%{
Load the data. Each row of the sheet gives the date of a policy meeting or non-meeting intervention. Each column gives the change in the yield on a 
particular asset from the previous trading day's market close to the current trading day's market close.

%The |xlsread| function places numeric data into its first output variable
and a cell array containing any text information into its second output
array. This will contain the series' names (in its first row) and the dates
of the policy meetings and non-meeting interventions.
%}
[numericData,cellArrayOfStringData]=xlsread(fileName,sheetName); 
%Create a vector of serial date numbers for the observations
serialDateNumbersForInterventions=datenum(char(cellArrayOfStringData{2:end,1}),'mm/dd/yyyy');
%Put the variable names into their own cell array and then clear
%|cellArrayOfStringData|.
variableNames=cellArrayOfStringData(1,2:end); 
clear cellArrayOfStringData;

%We are now prepared to identify the sample observations of interest. 
assetPriceChanges=numericData(serialDateNumbersForInterventions>=startDate & serialDateNumbersForInterventions<=endDate,:);
theseDates=serialDateNumbersForInterventions(serialDateNumbersForInterventions>=startDate & serialDateNumbersForInterventions<=endDate);

%Remove March 18, 2009 from the sample.
QE1date=datenum([2009 3 18]);
assetPriceChanges=assetPriceChanges(theseDates~=QE1date,:);
theseDates=theseDates(theseDates~=QE1date,:);

%% *Part III: Obtain factors, orthogonalize, rotate and re-scale* 

%% 4.Factors Obtain factors with princomp, standardize, compute loadings 
[inverseFactorLoadings,estimatedFactors,factorVariances]=princomp(zscore(assetPriceChanges));
%The inverse factor loadings and estimated factors require two
%modifications before proceeding. First, we need to standardize the factors
%to have unit variance. Therefore, the factor loadings should be premultiplied
%by the factors' standard deviations. We perform this multiplication as
%well as the inversion of |inverseFactorLoadings| in one step.
factorLoadings=diag(sqrt(factorVariances))/inverseFactorLoadings;
estimatedFactors=estimatedFactors/diag(sqrt(factorVariances));

%Next, we need to calculate the factor loadings for the original asset
%price changes rather than for their standardized values. For this, we
%postmultiply the factor loadings by the diagonal matrix of asset price
%changes' standard deviations.
factorLoadings=factorLoadings*diag(std(assetPriceChanges));

%Following GSS, we focus on the first two principle components. To adhere
%to their notation, we put these two components in |F| and the
%corresponding factor loadings in |gamma|.
gamma=factorLoadings(1:2,:);
F=estimatedFactors(:,1:2);

%% 5. \section{Factor Rotation}
%{
Folowing \cite{GSS}, we wish to transform the estimated factors to give them a
structural interpretation. Let $F$ denote the $1 \times 2$ vector of
factors, and let $U$ be a $2\times 2$ matrix. The $ 1 \times 2$ vector of transformed factors $Z$ satisfies
 \[ Z = F U \]

From the definition of a rotation, $U$ is orothonormal. That is, $U'U=I$. This guarantees 
\[\mathbb{E}[Z'Z] = \mathbb{E}[U'F'FU]=U'\mathbb{E}[F'F]U=U'U=I\] 

That is, the transformed factors are orthogonal and have variances
identical to the original factors. 

\subsection{Identifying Restrictions}
We use four algebraic restrictions and two sign restrictions to choose the four elements of $U$. 
The definition of a rotation gives us three restrictions, which we express explicitly by writing
\[ U = \left[\begin{array}{cc}\alpha_1 & \beta_1 \\ \alpha_2 & \beta_2
\end{array}\right]\]
In this notation, orthonormality requires
\begin{eqnarray*}

    \alpha_1^2+\alpha_2^2 & = & 1, \\
    \beta_1^2+\beta_2^2 & = & 1,\\
    \alpha_1\beta_1+\alpha_2\beta_2 & = & 0.

\end{eqnarray*}

For the fourth algebraic restriction, we require that a change in the
second element of $Z$ has no impact on the contemporaneous interest rate.
For this, we write
\[ R = F\Gamma =Z U^{-1} \Gamma. \]
Here, $R$ is the $1\times k$ vector of interest rates and $\Gamma$ is the
$2 \times k$ matrix of factor loadings. Without loss of generality, we
place the contemporaneous interest rate into the first element of $R$; and we
denote the $i,j$ element of $\Gamma$ with $\gamma_{ij}$. Then,
$\gamma_{11}$ gives the response of the contemporaneous interest rate to
the first factor in $F$ and $\gamma_{21} gives the response to the second
factor. 

To calculate $\partial R_1/\partial Z_2$, we first obtain 
\[ U^{-1} = \left[\begin{array} \beta_2 & -\beta_1 \\ -alpha_2 & \alpha_1
\right]\end{array}\cdot \frac{1}{\alpha_1\beta_2-\alpha_2\beta_1} \]

Using this to write $F=ZU^{-1}$ line-by-line gives us

\begin{eqnarray*}

    F_1 & = & \frac{Z_1\beta_2 - \alpha_2
    Z_2}{\alpha_1\beta_2-\alpha_2\beta_1} \\

    F_2 & = & \frac{-\beta_1 Z_1 + \alpha_1
    Z_2}{\alpha_1\beta_2-\alpha_2\beta_1} 

\end{eqnarray*}

Finally, plugging this into the first equation in $R = F\Gamma$ gives us
\[
R_1 = \frac{\gamma_{11}(Z_1\beta_2 - \alpha_2
    Z_2) + \gamma_{21}(-\beta_1 Z_1 + \alpha_1
    Z_2)}{\alpha_1\beta_2-\alpha_2\beta_1}
\]

Therefore,
\[ \frac{\partial R_1}{\partial Z_2} =
\frac{\alpha_1\gamma_{21}-\alpha_2\gamma_{11}}{\alpha_1\beta_2-\alpha_2\beta_1}
\]
Setting the numerator of this fraction to zero requires that
\[\frac{\alpha_1}{\alpha_2}=\frac{\gamma_{11}}{\gamma_{21}}\]
This is our fourth restriction.

Although we have four restrictions and four unknowns, these restrictions
alone are not enough to identify $U$, because we could multiply $U$ by any
diagonal matrix $S$ with its diagonal elements equal to either $1$ or $-1$
and the resulting product would also satisfy these four algebraic restrictions. Put
differently, we require two sign restrictions. We use
\begin{eqnarray*}
 \frac{\partial R_1}{\partial Z_1} & \geq & 0 \\
 \frac{\partial R_2}{\partial Z_2} & \geq & 0 \\
\end{eqnarray*}

To determine the first restriction's implications for the elements of $U$,
use the expression for $R_1$ above to get 
\[
\frac{\partial R_1}{\partial Z_1} =
\frac{\beta_2\gamma_{11}-\beta_1\gamma{21}}{\alpha_1\beta_2-\alpha_2\beta_1}
\]
To determine the derivative's sign, we pull $\alpha_1\beta_2$ out of the
denominator and $\beta_2\gamma_{11}$ out of the numerator. Then, using the
restriction that $\alpha_1/\alpha_2=-\beta_2/\beta_1 =
\gamma_{11}/\gamma{21}$, we get
\begin{eqnarray*}
\frac{\partial R_1}{\partial Z_1} & = &
\frac{\beta_2\gamma_{11}}{\alpha_1\beta_2}\frac{1-\frac{\beta_1\gamma_{21}}{\beta_2\gamma_{11}}}{1-\frac{\alpha_2\beta_1}{\alpha_1\beta_2}}
\\
& = &
\frac{\gamma_{11}}{\alpha_1}\frac{1-\frac{\beta_1\alpha_2}{\beta_2\alpha_1}}{1-\frac{\beta_1\alpha_2}{\beta_2\alpha_1}}
\\
& = & \frac{\gamma_{11}{\alpha_1}
\end{eqnarray*}
Therefore, this restriction requires os to chooose $\alpha_1$ to have the
same sign as $\gamma_{11}$.

The second restriction uses
\[
\frac{\partial R_2}{\partial Z_2} =
\frac{\alpha_1\gamma_{22}-alpha_2\gamma{12}}{\alpha_1\beta_2-\alpha_2\beta_1}
\]
Pulling $\alpha_1$ out of the numerator and $\alpha_1\beta_2$ out of the
denominator yields
\begin{eqnarray*}\frac{\partial R_2}{\partial Z_2} & = &
\frac{\gamma_{22}-\frac{\alpha_2}{\alpha_1}\gamma_{12}}{\beta_2\left(\1-\frac{\alpha_2\beta_1}{\alpha_1\beta_2}\right)}
\\
& = &
\frac{\gamma_{22}-\frac{\gamma_{21}}{\gamma_{11}}\gamma_{12}}{\beta_2\left(1+\left(\frac{\gamma_{21}}{\gamma_{11}}\right)^2\right)}\
\end{eqnarray*}
So we wish to set the sign of $\beta_2$ equal to the sign of
$\gamma_{22}-\gamma_{12}\gamma_{21}/\gamma_{11}$. 

%}

%{
\subsection{Calculating the Identified Rotation}
Given these restrictions, the calculation of the identified rotation
proceeds straightforwardly. Squaring the zero-impact restriction on
$\alpha_1/\alpha_2$, using it to replace $\alpha_2^2$ in the first vector
magnitude restriction, and imposing the first sign constraint gives us.
\[ 
\alpha_1 = \textrm{sign}(\gamma_{11})\frac{1}{\sqrt{1+\left(\frac{\gamma_21}{\gamma{11}}\right)^2}
\]
Plugging this into the zero-impact restriction gives us $\alpha_2$.
%}
alpha_1 = sign(gamma(1,1))/sqrt(1+(gamma(2,1)/gamma(1,1))^2);
alpha_2 = alpha_1*gamma(2,1)/gamma(1,1);
%{
The calculation of $\beta_1$ and $\beta_2$ proceeds analogously. Squaring
the orthogonality restriction, using it to replace $\beta_1$ in the second
vector magnitude restriction, and imposing the second sign restriction
gives us
\begin{eqnarray*}
\beta_2 & = &
\textrm{sign}(\gamma_{22}-\frac{\gamma_{21}}{\gamma_{11}}\gamma_{12})\frac{1}{\sqrt{1+\left(\frac{\alpha_2}{\alpha_1}\right)^2}}
\\
& = & 
\textrm{sign}(\gamma_{22}-\frac{\gamma_{21}}{\gamma_{11}}\gamma_{12})\frac{1}{\sqrt{1+\left(\frac{\gamma_{21}}{\gamma_{11}}\right)^2}}
\\
& = &
\textrm{sign}(\gamma_11})\textrm{sign}(\gamma_{22}-\frac{\gamma_{21}}{\gamma_{11}}\gamma_{12})
\alpha_1 \\
& = & \textrm{sign}(\gamma_{11}\gamma_{22}-\gamma{12}\gamma{21})\alpha_1
\end{eqnarray*}
Plugging this into the orthogonality restriction gives us $\beta_1$. With
this in place, we may assemble $U$ and calculate the transformed factors.
%}
beta_2 = sign(det(gamma(1:2,1:2)))*alpha_1;
beta_1 = -beta_2*gamma(2,1)/gamma(1,1);
U = [alpha_1 beta_1; alpha_2 beta_2];
Z=F*U;

%{
\section{Factor Scaling}
As calculated, the variances of the factors equal the largest eigenvalues
of the data's variance-covaraiance matrix. Since this matrix is scale-free,
the variances have no direct economic interpretation. Therefore, it is
helpful to scale them to have a known impact on the data. For this, we
choose a $2\times 2$ diagonal matrix $S$ and define $Z^\star=ZS$. From the
equation for $R$, we get
\begin{eqnarray*}
R & = & Z S S^{-1} U^{-1}\Gamma \\
  & = & Z^\star S^{-1}U^{-1}\Gamma \\
  & = & Z^\star \Gamma^\star
\end{eqnarray*}

We choose the diagonal elements of $S$ so that
\begin{eqnarray*}
 \frac{\partial R_1}{\partial Z^\star_1} & = & 1 \\
 \frac{\partial R_5}{\partial Z^\star_2} & = & 1
\end{eqnarray*}
Since the two factors will be used as regressors elsewhere, these unit-impact
restrictions will ease the interpretation of the resulting regression
coefficients.

To implement these scales, we proceed in two steps. \\
\smallskip
1. We first compute normalized factors  $ \hat{Z}$ and  loadings $ \hat{\Gamma}$ such that
\begin{eqnarray*}
 \frac{\partial R_1}{\partial \hat{Z}_1} & = & 1 \\
 \frac{\partial R_2}{\partial \hat{Z}_2} & = & 1
\end{eqnarray*}
by appropriately choosing the diagonal elements of an initial diagonal matrix $ \hat{S} $ .


From the calculations of the sign restrictions, we know that the unique
values of $\hat{S}_{11}$ and $\hat{S}_{22}$ that satisfy the unit-impact restrictions
are
\begin{eqnarray*}
\hat{S}_{11} & = & \frac{\gamma_{11}}{\alpha_1} \\
\hat{S}_{22} & = &
\frac{\gamma_{22}-\frac{\gamma_{21}}{\gamma_{11}}\gamma_{12}}{\beta_2\left(1+\left(\frac{\gamma_{21}}{\gamma_{11}}\right)^2\right)}\\
\end{eqnarray*}

It follows that
\begin{eqnarray*}
\hat{Z}& = & Z \hat{S} \\
\hat{\Gamma}& = & \hat{S}^{-1}U^{-1}\Gamma \
\end{eqnarray*}

\smallskip

2. We then choose a diagonal matrix $ S $ such that

\begin{eqnarray*}
S_{11}& = & 1 \\
S_{22}& = & \hat{\Gamma}_{25}
\end{eqnarray*}.

And define
\begin{eqnarray*}
Z^\star & = & \hat{Z} S \\
\Gamma^\star & = & S^{-1}\hat{\Gamma }
\end{eqnarray*}

Hence $S$ scales the factors and loadings such that the effect of the second factor on the fifth interest rate is unity, as desired.

%}
Shat=eye(2);
Shat(1,1)=gamma(1,1)/alpha_1;
Shat(2,2)=det(gamma(1:2,1:2))/(gamma(1,1)*beta_2*(1+(gamma(2,1)/gamma(1,1))^2));
Zhat = Z*Shat;
gammaStarhat=Shat\(U\gamma);

S=eye(2); 
S(2,2)=gammaStarhat(2,5); 
SInverse=eye(2); 
SInverse(2,2)=1/gammaStarhat(2,5); 

gammaStar=SInverse*gammaStarhat; 
Zstar    =Zhat*S; 

clear gammaStarhat Zhat Shat; 

%{
\section{Measures of Fit}
To analyze how well the first two principle components explain the asset
price data, we calculate the fitted values implied by the two factors as
well as the residuals.


%}
targetFactor=Zstar(:,1);
pathFactor=Zstar(:,2);

fittedValues=Zstar*gammaStar;

targetFactorPrices = targetFactor*gammaStar(1,:);
pathFactorPrices = pathFactor*gammaStar(2,:);

measuresOfFit=zeros(size(fittedValues,2),3);
measuresOfFit(:,1)=(var(targetFactorPrices)./var(assetPriceChanges))';
measuresOfFit(:,2)=(var(pathFactorPrices)./var(assetPriceChanges))';
measuresOfFit(:,3)=1-(var(fittedValues)./var(assetPriceChanges))';

%{
\section{First and Second Moments}

We now measure the five ``idiosyncratic'' errors. The next table reports
the means and standard deviations of these along with the two factor, all
in basis points.
%}
idiosyncraticErrors=assetPriceChanges-fittedValues;
means=mean(100*[targetFactor pathFactor idiosyncraticErrors]);
standardDeviations=std(100*[targetFactor pathFactor idiosyncraticErrors]);
moments=[means;standardDeviations];

%The last table reported gives the variance-covariance matrix of the two
%factors and the idiosyncratic errors. This serves two purposes. First, it
%verifies the orthogonality between the factors and the errors (and each
%other), second, it gives us a sense of how far off the assumption of
%independent idiosyncratic errors really is. 
correlations=corr([targetFactor pathFactor idiosyncraticErrors]);


%{
\section{Spreadsheet Output}

For each sample, we write the factors, the idiosyncratic errors, and their
summary statistics to a spreadsheet. Each of these is named for its sample.
%}
spreadsheetName=['sample' num2str(caseSample) 'Results'];
%
%We begin by writing the factors to one sheet. Its header describes the
%data and labels the columns.
xlswrite(spreadsheetName,{'Estimated Factors in Basis Points'},'Factors','B1');
xlswrite(spreadsheetName,{'Date','Target','Path'},'Factors','A2');
%Next, insert the column of dates.
dateStrings=cellstr(datestr(theseDates));
xlswrite(spreadsheetName,dateStrings,'Factors','A3');
%Finally, we insert the data itself.
xlswrite(spreadsheetName,[100*targetFactor 100*pathFactor],'Factors','B3');

%The next sheet in the file contains the idiosyncratic errors. As with the
%sheet containing the factors, we give it a header and put dates in
%the first column.

xlswrite(spreadsheetName,{'Idiosyncratic Errors in Basis Points'},'Errors','B1');
columnLabels=cell(1,1+size(idiosyncraticErrors,2));
columnLables{1,1}='Date';
for j=2:size(idiosyncraticErrors,2)+1
    columnLabels{1,j}=['ID' num2str(j-1)];
end
xlswrite(spreadsheetName,columnLabels,'Errors','A2');
xlswrite(spreadsheetName,dateStrings,'Errors','A3');
xlswrite(spreadsheetName,100*idiosyncraticErrors,'Errors','B3');

%The file's final sheet contains the moments and measures of fit.
%The informational header gives the source file for the data, the source
%sheet name for the data, the sample period used, and the number of
%resulting observations.
header=cell(5,2); 
header(1,:)={'Source Data Spreadsheet File:',fileName}; 
header(2,:)={'Source Data Sheet:',sheetName}; 
header(3,:)={'Sample Period Start:',datestr(startDate)}; 
header(4,:)={'Sample Period End:',datestr(endDate)};
header(5,:)={'Number of Observations:',size(targetFactor,1)}; 

xlswrite(spreadsheetName,header,'Moments','A1');

%The next panel reports the measures of fit.
xlswrite(spreadsheetName,{'Percentage of Variance From'},'Moments','B7');
xlswrite(spreadsheetName,{'Target Factor','Path Factor','Idiosyncratic Error'},'Moments','B8');

variableNames=cell(size(assetPriceChanges,2),1);
for j=1:size(assetPriceChanges,2)
   variableNames{j,1}=['R' num2str(j)];
end
xlswrite(spreadsheetName,variableNames,'Moments','A9');
xlswrite(spreadsheetName,100*measuresOfFit,'Moments','B9');
    
%The next panel reports the first and second moments.    
verticalOffset=10+size(assetPriceChanges,2);
targetCell=['B' num2str(verticalOffset)];
xlswrite(spreadsheetName,{'First and Second Moments (in Basis Points) of'},'Moments',targetCell);
columnLabels=cell(1,2+size(idiosyncraticErrors,2));
columnLabels{1,1}='Target Factor';
columnLabels{1,2}='Path Factor';
for j=3:2+size(idiosyncraticErrors,2)
   columnLabels{1,j}=['ID' num2str(j-2)]; 
end
verticalOffset=verticalOffset+1;
targetCell=['B' num2str(verticalOffset)];
xlswrite(spreadsheetName,columnLabels,'Moments',targetCell);
targetCell=['A' num2str(verticalOffset)];
xlswrite(spreadsheetName,{'';'Mean'; 'Standard Deviation'},'Moments',targetCell);
verticalOffset=verticalOffset+1;
targetCell=['B' num2str(verticalOffset)];
xlswrite(spreadsheetName,moments,'Moments',targetCell);

%The final panel reports the correlations.
verticalOffset=verticalOffset+2+size(moments,1);
targetCell=['A' num2str(verticalOffset)];
xlswrite(spreadsheetName,{'Correlations'},'Moments',targetCell);
verticalOffset=verticalOffset+1;
targetCell=['B' num2str(verticalOffset)];
xlswrite(spreadsheetName,columnLabels,'Moments',targetCell);
verticalOffset=verticalOffset+1;
targetCell=['A' num2str(verticalOffset)];
xlswrite(spreadsheetName,columnLabels','Moments',targetCell);
targetCell=['B' num2str(verticalOffset)];
xlswrite(spreadsheetName,correlations,'Moments',targetCell);

end
