% =========================================================================
% DSGEMain.m 
% Description 
% 
% 1. Load the mode and data which generated the original (baseline) forecast, which is
% stored in the mat-file BrookingsMain 
%
% 2. Load a different mode, with possibly different transition matrices 
%    or specify a set of shocks to hit a target, e.g. higher long run
%    inflation. 
%    The code below considers the two experiments reported in the paper: 
%
%    *caseExperiment==1*, Higher Expected Inflation
%    *caseExperiment==2*, Shorter Deleveraging, i.e. lower rhob. 
%
% 3. Conditional on the changes in 2, find set of shocks that will keep 
%    the FFR at the same path as the original forecast. 
% 
% 4. Generate a set of new conditional forecasts for the main variables.  
% 
% 5. Trace out the baseline and new conditional forecast in the
% inflation & unemployment space 
%
% These codes are provided as part of replication files for 
% "Macroeconomic Effects of FOMC Forward Guidance" by CEFJ. 
%
% NOTE: Data was recently updated so may not exactly match the final
% graphs in the paper. 
% =========================================================================
clear all; close all; 
cucd=pwd; 
clc; 
cd .. 
addpath(genpath([cd,'\utilities']))
cd(cucd); 
caseExperiment=1; 
loadFolder=cucd;

% Decide which experiment to run
if caseExperiment==1    
%% Experiment 1: Higher Expected Inflation Experiment 
settingsFolder=fullfile(loadFolder,'LR Expected PI 1');
elseif caseExperiment==2 
%% Experiment 2: Lower Persistence of RHOB 
settingsFolder=fullfile(loadFolder,'Shorter Deleveraging');
else 
    error('caseExperiment must be 1 or 2') 
end 

settingsName='stateCountSettings'; 
add2sol.NLeadsFFRobs=10; 

% =========================================================================
%% End User Input
% =========================================================================
%% 1. Load the output of the baseline forecast
cd(loadFolder); 
load BrookingsWorkspace; 
clear forc_mat2;
parnames=coefnames; clear coefnames; 
sampleStru=sampleSS; clear sampleSS; 
forcBase=forc_mat; clear forc_mat; 
cd(cucd); 

cd(settingsFolder);
disp('Loading Settings File');
eval(settingsName);
cd(cucd);

%% 2. Solve Original Model and Extract Names
[GG, RR, CC, eu, SDX, ZZ, initss, ssvec,~, ssNames,stateNames,shockNamesStru]...
    =feval(funcmod,param,solveopt,addsol);

%% 3. Define nameStru, Structure with the Names 
shockNames=shockNamesStru.long; 
[Znames,ZposInStates]=extractObsNames(ZZ(:,:,1),stateNames);
nameStru.stateNames=stateNames;
nameStru.shockNames=shockNames;
nameStru.obsNames=Znames;
shockNames=shockNamesStru.long; 
posStates=cellposition(counter.names,stateNames); 
sampleVec=sampleStru.sampleVec; 

%% 4. Calculate Constants to be used in the series defined in Counter 
posFFR=cellposition('FFR',stateNames);
paramFFR=CC(posFFR);
paramInflation=param( cellposition('pi^{ss}',parnames) );
paramPCECore=CC( cellposition('PCE Core',stateNames) );
paramHours=CC( cellposition('Hours',stateNames) );
paramGAM100=param(cellposition('gamma_{\\star_{100}}',parnames) );
paramGDPDef=CC( cellposition('GDP deflator',stateNames) );
paramGDP=paramGAM100+paramInflation-paramGDPDef;
paramCONS=paramGAM100+( paramInflation-CC(cellposition('C deflator',stateNames)));
paramINV=paramGAM100+( paramInflation-CC(cellposition('Inv deflator',stateNames)));

%% 5. Fill IN SSVEC with those constants 
ssVec=zeros(size(counter.names)); 
ssVec(cellposition('GDP',counter.names))=paramGDP; 
ssVec(cellposition('Consumption',counter.names)) = paramCONS;
ssVec(cellposition('Investment',counter.names))= paramINV;
ssVec(cellposition('FFR',counter.names))= paramFFR;
ssVec(cellposition('Hours',counter.names))= paramHours;
ssVec(cellposition('PCE Core',counter.names))= paramPCECore;
temp=regexp(counter.names(:),'FFR'); 
tempos=zeros(length(temp),1); 
for ii=1:length(tempos);
    if ~isempty(temp{ii})
    tempos(ii)=ii;
    end
end 
ssVec(tempos~=0)=paramFFR; 
printcell([counter.names(:) num2cprec([ssVec(:) counter.scale(:)]) ]); 
%quer('c'); 
clear temp*; 


%% 6. Settings for Minforc, which will minimize the SSR with the FFR Path 

%% 6.a Positions of shocks and signals that will be shut down 
if minforc.signalEnd < addsol.NLeadsFFRObs
    minforc.signalShut=(minforc.signalEnd+1:addsol.NLeadsFFRObs);
    minforc.loadShut  =zeros(length(minforc.signalShut),1);
    minforc.shockShut =zeros(length(minforc.signalShut),1);
    minforc.signalShutPos=[];
    for ii=1:length(minforc.signalShut);
        tempstr=strfind(stateNames,['Signal ',num2str(minforc.signalShut(ii))]);
        tempStru=findNonEmptyCell(tempstr);
        minforc.signalShutPos=[minforc.signalShutPos;tempStru.posNonEmpty(:)];
        minforc.loadShut(ii)=cellposition(['mathcal{B}^{S2}_{',num2str(minforc.signalShut(ii)),'}'],parnames);
        minforc.shockShut(ii)=cellposition(['ID Signal ',num2str(minforc.signalShut(ii))],shockNames);
    end
else 
    minforc.signalShut=[]; 
    minforc.loadShut=[]; 
    minforc.shockShut=[]; 
    minforc.signalShutPos=[]; 
end 

%% 6.b Determine which shocks are On and Off after adding 
% the deleted signals, signalShut, to shockShutAddPos. 
minforc.Nx=size(SDX,1);
if isempty(minforc.shockShutAdd)==0
    minforc.shockShutAddPos=cellposition(minforc.shockShutAdd,shockNames);
else
    minforc.shockShutAddPos=[];
end

if isempty(minforc.shockShutAddPos)==false || isempty(minforc.shockShut)==false; 
    minforc.shockOnPos=setdiff( (1:minforc.Nx)', [minforc.shockShutAddPos(:);minforc.shockShut(:)] );
    minforc.shockOn=shockNames(minforc.shockOnPos);
    minforc.shockOffPos=[minforc.shockShutAddPos(:);minforc.shockShut(:)]; 
    minforc.shockOff=shockNames(minforc.shockOffPos);
else 
    minforc.shockOn=(1:minforc.Nx)'; 
    minforc.shockOn=shockNames; 
    minforc.shockOffPos=[]; 
    minforc.shockOff=[]; 
end 
dispaj('Number of shocks Active = ',length(minforc.shockOn),' out of ',minforc.Nx); 

if isempty(minforc.shockOff)==false;
    disp('Shocks off');    
    printcell(minforc.shockOff);
else
    disp('No Shocks Off');
end
%quer('c');

%% 7. Load a parameter value (optional) or Keep the Existing One 
if isempty(modeFileName)==false;   
    cd(modePath);    
    if exist('modeSheetName','var')==0 || isempty(modeSheetName)
        dispaj('Loading from ',modeFileName);
        [parcounter,parcounterNames]=xlsread(modeFileName);
    else
        dispaj('Loading from ',modeFileName,', Sheet ',modeSheetName);
        [parcounter,parcounterNames]=xlsread(modeFileName,modeSheetName);
    end
    cd(cucd);
    flagStruct.loadedMode=1; 
    if size(parcounterNames,1)-size(parcounter,1) == 1
        disp('Using Top Row for ModeNames');
        minforc.altName=parcounterNames(1,2:end);
        parcounterNames=parcounterNames(2:end,1);
        if size(parcounterNames,1)-size(parcounter,1) ~=0
            error('parcounterNames can exceed parcounter by 1 row only')
        end
    else
        error('Please provide ModeNames');
    end
    counter.param=parcounter;
    counter.parnames=parcounterNames;
    clear parcounter parcounternames;
    
    disp('Differences in Parameters');
    pardiffPos=find( abs(param-counter.param) > 0.001 );
    printcell( [parnames(pardiffPos) num2cprec( [param(pardiffPos) counter.param(pardiffPos)] ) ] );
else 
    disp('No New Mode loaded');
    flagStruct.loadedMode=0;
    counter.param=param; 
    counter.parnames=parnames; 
    if isfield(minforc,'altName')==false ||  isempty(minforc.altName)==true 
        error('Please define minforc.altName if not loading a new Mode'); 
    end 
end 

%% 9. Check Loaded Forecasts Match those loaded in the workspace (baseline)
[forcCheck]=zb_quickforecast(forc_pos,GG(:,:,end),ZZ(:,:,end),CC(:,end),nforc-1,...
    KFStru.smoothSt(end,:),pos_gdpdef,popF,in); 
disp('Comparing Matrix of Forecasts'); 
maxdif=comparemat(forcBase(2:end,:),forcCheck); 
if maxdif > 1e-5 
    error('Forecasts do not coincide'); 
end 
clear forcCheck maxdif; 

%% 10. Solve model with alternative mode, if one has been loaded 
if flagStruct.loadedMode==1;
    if isempty(counter.funcmod)==true
        counter.funcmod=funcmod;
    end
    counter.addsol=addsol;
    counter.addsol.NLeadsFFRObs=add2sol.NLeadsFFRobs;
    [GCount, RCount, CCount, euCount, SDXCount, ZCount, ~,~,~, ~,stateNamesCount,shockNamesStruCount]...
        =feval(counter.funcmod,counter.param,solveopt,counter.addsol);    
    minforc.G=GCount(:,:,end);
    minforc.R=RCount(:,:,end);
    minforc.Z=ZCount(:,:,end);
    minforc.C=CCount(:,end);
    minforc.SDX=SDXCount(:,:,end); 
    clear GCount RCount CCount euCount SDXCount ZCound 
else 
    minforc.G=GG(:,:,end);
    minforc.R=RR(:,:,end);
    minforc.Z=ZZ(:,:,end);
    minforc.C=CC(:,end);
    minforc.SDX=SDX(:,:,end); 
end 

%% 11. Additional Settings for Minforc 

%% 11.a Dimensions 
minforc.Nmatch=length(minforc.matchNames); 
minforc.Nfree=length(minforc.shocksFree); 
minforc.Nforc=length(sampfor)-1; 
minforc.Nx=size(SDX,1); 

%% 11.b Position to launch forecast
% minforc.dateJumpPos=find(sampleStru.sampleVec==minforc.dateJump); 
% dispaj('Jumping Point for the Forecast:',sampleVec(minforc.dateJumpPos)); 
% if isempty(minforc.dateJumpPos)==true 
%     error('Cannot find date at which to match the FFR Path'); 
% end 

%% 11.c Check if there is available data, i.e. smooth shocks will be part
%% of the forecast 
% if minforc.dateJumpPos < length(sampleStru.sampleVec) 
%     minforc.availDataPos=(minforc.dateJumpPos+1:length(sampleVec) ); 
%     disp('Available Data: '); 
%     printcell( sampleVec(minforc.dateJumpPos+1:end) );     
% else 
%     minforc.availDataPos=[]; 
%     disp('No Available Data'); 
% end         
%quer('c'); 

%% 12. Define Data, states and shocks used in the forecast 
% Recall minforc.dateJump is T, date at which the forecast starts. 
% If T is not the end of the sample, then will assign the smooth
% innovations to eta(T+1:Tend), zeros otherwise. 
% Else eta(T+1:Fend) will be zeros
minforc.matchPos=cellposition(minforc.matchNames,stateNames); 
minforc.innovPos=cellposition(minforc.shocksFree,shockNames); 
minforc.data=Y(end,:)'; 
%% 12.a State at forecast launch date 
minforc.sTMinusOne=KFStru.smoothSt(end,:)'; 
%% 12.b Innovations at launch date 
minforc.innovMat=zeros(minforc.Nforc,minforc.Nx); 
innovMatBase=zeros(minforc.Nforc,minforc.Nx); 

%% 13. Baseline Non-Adjusted Forecast
disp('If Forecast Data is before break, this is not correct'); 
[stforcBase,yforcBase]=forecastLoop(GG,RR,ZZ,CC,ones(minforc.Nforc,1)*addsol.tauVec(end),...
    KFStru.smoothSt(end,:)',innovMatBase,minforc.Nforc); 

%% 14. Determine Target states that must match in the Alternative Forecast 
minforc.matchTarget=stforcBase(1,minforc.matchPos); 
tempStru=findNonEmptyCell(minforc.matchVal);
if isempty(tempStru.nonEmpty)==false
    disp('Replacing forecast state with a target'); 
    minforc.matchTarget( tempStru.posNonEmpty ) = tempStru.nonEmpty ; 
end 
disp('Targets to Match'); 
printcell( [minforc.matchNames(:) num2cprec( minforc.matchTarget(:) ) ] ); 
%quer('c'); 

%% 15.a Shut down signals 
% If the Number of Signals is less than available, shut down extra
%% signals 
if isempty(minforc.signalShutPos)==false;
    minforc.sT(minforc.signalShutPos)=0;
end

%% 15.b Shut down additional shocks 
if isempty(minforc.shockShut)==false;
    minforc.innovMat(:,minforc.shockShut)=0;
end

%% 15.c innovMatOrig: matrix of innovations to be used in the optimization 
minforc.innovMatOrig=minforc.innovMat; 

%% 16. saveFolder 
if isempty(modeAddSubfolder)==true
    saveFolder=cr_dir(settingsFolder,[char(minforc.altName),' ',strdate]);
else
    saveFolder=cr_dir(settingsFolder,[char(minforc.altName),' ',modeAddSubfolder,' ',strdate]);
end
subtitleString=stringDifference(saveFolder,cd,'\');

if flagStruct.loadedMode==1
    cd(saveFolder);
    xlswrite(modeFileName,[parnames num2cprec(counter.param,10)]);
    cd(cucd);
end


cd(saveFolder); 
save workspace; 
cd(cucd); 

minforc.tauVecFake=ones(minforc.Nforc,1); 
minforc.forcPos=cellposition(forc_names,stateNames); 
minforc.inStru=in; 
minforc.inStru.forc_pos=minforc.forcPos; 
minforc.popForecast=pop_gr( length(sample)+1:end ); 

%% 17.a Initial Guess and First Evaluation of the function 
minforc.innovT=minforc.innovMat(1,:);
x0=minforc.innovT(minforc.innovPos); 
fval=minStateDiscrep(x0,minforc,minforc.innovT); 


%%
%% 17.b Set the structure minProblem with the options for the optimization 
minproblem.optimset=optimset('MaxIter',1000,'FunValCheck','On','Display','Iter'); 
minproblem.maxIter=10000; 
minproblem.TolFun=1e-5; 
minproblem.TolX=1e-5; 
minproblem.options=optimset('MaxIter',minproblem.maxIter,'FunValCheck','On',...
    'TolFun',minproblem.TolFun,...
    'TolX',minproblem.TolX); 
minproblem.solver='lsqnonlin'; 

%% 18. Minimization and conditional forecasts (forcOut)
% .forcAlt: Alternative ,adjusted, forecast. Compare to forcBase; 
% .stF: state forecast 
% .yF: observable forecast 
% .etaMat: matrix used to generate forecast 
% .min: output from the minimization 
minforc.flagSimul=0; 
minforc.flagVerbose=1; 

cd(saveFolder); 
save workspace; 
cd(cucd); 

minforc=orderfields(minforc); 

forcOut=forcStateSub(minforc,minproblem); 
forcAlt=zeros( size(forcBase) ); 
forcAlt(1,:)=forcBase(1,:); 
forcAlt(2:end,:)=forcOut.forcAlt; 

forcStru.pos=in.forc_pos; 
forcStru.addp=in.forc_addp; 
forcStru.glabels=in.forc_glabels; 
forcStru.names=in.forc_names; 
forcStru.consAdj=forc_cons; 
forcStru.consNonAdj=in.consnoadj; 
forcStru.scale=in.forc_scale; 
forcStru.base=forcBase; 
forcStru.alt=forcAlt; 
forcStru.minforc.altName=minforc.altName; 

cd(saveFolder); 
save workspace; 
cd(cucd); 

%% 19. Generate Two additional forecasts for Comparison 

%% 19.a forcAltBaseSh: Alternative mode, with Base shocks 
if flagStruct.loadedMode==1    
    stAltBaseSh=forecastLoop(minforc.G,minforc.R,minforc.Z,minforc.C,...
        minforc.tauVecFake,minforc.sTMinusOne(:),innovMatBase,minforc.Nforc);
    forcAltBaseSh=zeros(minforc.Nforc+1,length(forc_pos));
    forcAltBaseSh(1,:)=forcBase(1,:);
    forcAltBaseSh(2:end,:)=zb_quickforecast_sub(stAltBaseSh,minforc.popForecast,minforc.inStru);
    clear stAltBaseSh;
else 
    disp('Alternative and Baseline Model Identical') 
end 

%% 20. Extract Positions to Plot 
posStates=cellposition(counter.names,stateNames);
if flagStruct.loadedMode==1   
    counter.posStatesAll=cellposition(stateNamesCount,stateNames);
    counter.posInovAll=cellposition(shockNamesStruCount.long,shockNames);
    counter.pos=cellposition(counter.names,stateNamesCount);
    if isequal(counter.pos,posStates)==false 
        error('Code must be ammended when States Differ in Position across Models') 
    end 
    printcell([stateNamesCount(counter.pos) stateNames(posStates)]);
else
    counter.posStatesAll=cellposition(stateNames,stateNames);
    counter.posInovAll=cellposition(shockNames,shockNames);
    counter.pos=cellposition(counter.names,stateNames);
end

paperPosition=[0.75 0.5 7.5 10];

%% ========================================================================
%% 21.Plot forecasts 

%% Baseline vs. Alt 
tempHandle=zeros(2+flagStruct.loadedMode,1); 
tempHandle(1)=zb_plot2forecasts(forcBase,forcAlt,'Baseline vs Counter Enforcing the Observed FFR Path',...
    {'Baseline',char(minforc.altName)},sampfor,forc_glabels,saveFolder,'Baseline vs. Alt'); 

%% Baseline vs. AltBaseSh 
if flagStruct.loadedMode==1 
tempHandle(3)=zb_plot2forecasts(forcBase,forcAltBaseSh,'Baseline vs. Alternative with Baseline Shocks',...
    {'Baseline',[char(minforc.altName),' with Baseline Shocks']},...
    sampfor,forc_glabels,saveFolder,'Baseline and Alt w Baseline Shocks');        
end 
handleStruct.forecast=tempHandle; 
clear tempHandle; 

%% ========================================================================
%% 22. Projection to unemployment 
cd(cucd); 
loadURate; 
cd(cucd); 
urateStruct.start=2007.5; 
urateStruct.end  =sampleVec(end); 
urateStruct.startPos=find( urateStruct.sample==urateStruct.start ); 
urateStruct.endPos=find( urateStruct.sample==urateStruct.end ); 
urateStruct.refPos=cellposition('Hours',stateNames); 
temp=( find(sampleVec==urateStruct.start):find(sampleVec==urateStruct.end) ); 
urateStruct.refData=hoursData(temp); 
urateStruct.refN   =length( urateStruct.refData ); 
unemployment=urateStruct.data( urateStruct.startPos:urateStruct.endPos ); 
[urateCoff,urateRes]=ols(unemployment,[ones(urateStruct.refN,1) urateStruct.refData] ); 


%% 23. Adjust constant to hit an end-point unemployment target 
urateStruct.endTarget=8.7; 
urateStruct.insampleFit=urateCoff(1)*ones(urateStruct.refN,1)+urateCoff(2)*urateStruct.refData; 
urateStruct.consAdjustment=urateStruct.endTarget-urateStruct.insampleFit(end); 
urateCoff(1)=urateCoff(1)+urateStruct.consAdjustment; 

urateStruct.insampleFit=urateCoff(1)*ones(urateStruct.refN,1)+urateCoff(2)*urateStruct.refData; 
if abs( urateStruct.endTarget-urateStruct.insampleFit(end) ) > 0.00001 
    error('Did not hit the unemployment target') 
end 
corePCEForecast=zeros(minforc.Nforc+1,2); 
corePCEForecast(:,1)=forcBase(:,cellposition('PCE Core',forc_names) ); 
corePCEForecast(:,2)=forcAlt(:,cellposition('PCE Core',forc_names) ); 

urateForecast=zeros(minforc.Nforc+1,2); 
urateForecast(1,:)=urateStruct.insampleFit(end)*ones(1,2); 
urateForecast(2:end,1)=urateCoff(1)*ones(minforc.Nforc,1)+urateCoff(2)*stforcBase(:,urateStruct.refPos);
urateForecast(2:end,2)=urateCoff(1)*ones(minforc.Nforc,1)+urateCoff(2)*forcOut.stF(:,urateStruct.refPos);

%% 24 Plot in (U,Inflation Space) 
scat.colors=repmat([0.55 0.55 0.55],length(sampfor),1 ); 
scat.colors(end-1:end,:)=repmat([1 0 0],[2 1]); 
scat.colors(1,:)=[0 0 1]; 

brookings.xline=(5.2:0.1:6)'; 
brookings.yline=2*ones(length(brookings.xline),1); 

handleStruct.up=zeros(2,1); 
set(0,'DefaultAxesFontSize',11); 
for kk=1:2; 
    
    handleStruct.up(kk)=figure;
    set(gca,'FontSize',18);
    scatter(urateForecast(:,kk),corePCEForecast(:,kk),[],scat.colors,'filled','LineWidth',1.5);
        
    hold on;
    temphand=plot(urateForecast(:,kk),corePCEForecast(:,kk)); 
    set(temphand,'Color','k'); 
    
    temphand=plot(brookings.xline(:),brookings.yline); 
    set(temphand,'LineWidth',7,'Color',[0 0.8 0]); 
     
    temphand=plot(urateForecast(:,1),corePCEForecast(:,1)); 
    set(temphand,'LineWidth',1.5,'Color',[0.7 0.7 0.7]); 
                    
    yylim=limsaj( corePCEForecast(:,kk) );        
    yylim(1)=min(0,yylim(1)); 
    yylim(2)=max(3.5,yylim(2)); 
    ylim([yylim(1) yylim(2)]); 
    
    xxlim=limsaj( urateForecast(:,kk) );        
    xxlim(1)=min(5,xxlim(1)); 
    xxlim(2)=max(9,xxlim(2)); 
    xlim([xxlim(1) xxlim(2)]); 
       
    xlabel('Unemployment Rate');
    ylabel('PCE Core Inflation');
    set(gca,'Box','Off');
    hand1=hline(3);
    set(hand1,'LineStyle','--','Color','r');
    hand2=vline(7);
    set(hand2,'LineStyle','--','Color','r');
    switch kk 
        case 1 
            title('Baseline','FontSize',22); 
        case 2 
            title(minforc.altName,'FontSize',22); 
    end 
    set(handleStruct.up(kk),'PaperOrientation','landscape');
    set(handleStruct.up(kk),'PaperPosition',[0.2 0.1 10.5 8.3]);
    xlhand=set(gca,'xlabel');
    set(xlhand,'FontSize',17);
    ylhand=set(gca,'ylabel');
    set(xlhand,'FontSize',17);

end
set(0,'DefaultAxesFontSize','remove'); 
cd(saveFolder); 
saveas(handleStruct.up(1),'UDP Baseline','fig'); 
print(handleStruct.up(1),'-dpdf','UDP Baseline'); 
saveas(handleStruct.up(2),'UDP Alternative','fig'); 
print(handleStruct.up(2),'-dpdf','UDP Alternative'); 
cd(cucd);